<?php
/*========================================================================*\
|| ###################################################################### ||
|| # vBulletin 5.7.5 Patch Level 3 - Licence Number LD9934D570
|| # ------------------------------------------------------------------ # ||
|| # Copyright 2000-2025 MH Sub I, LLC dba vBulletin. All Rights Reserved.  # ||
|| # This file may not be redistributed in whole or significant part.   # ||
|| # ----------------- VBULLETIN IS NOT FREE SOFTWARE ----------------- # ||
|| # http://www.vbulletin.com | http://www.vbulletin.com/license.html   # ||
|| ###################################################################### ||
\*========================================================================*/

// note #1: arrays used by functions in this code are declared at the bottom of the page

/**
* Expand and collapse button labels
*/
define('EXPANDCODE', '&laquo; &raquo;');
define('COLLAPSECODE', '&raquo; &laquo;');

/**
* Size in rows of template editor <select>
*/
define('TEMPLATE_EDITOR_ROWS', 25);

global $vbphrase;

/**
* Initialize the IDs for colour preview boxes
*/
$numcolors = 0;

// #############################################################################
/**
* Checks the style id of a template item and works out if it is inherited or not
*
* @param	integer	Style ID from template record
*
* @return	string	CSS class name to use to display item
*/
function fetch_inherited_color($itemstyleid, $styleid)
{
	switch ($itemstyleid)
	{
		case $styleid: // customized in current style, or is master set
			if ($styleid == -1)
			{
				return 'col-g';
			}
			else
			{
				return 'col-c';
			}
		case -1: // inherited from master set
		case 0:
			return 'col-g';
		default: // inhertited from parent set
			return 'col-i';
	}

}

// #############################################################################
/**
* Saves the correct style parentlist to each style in the database
*/
function build_template_parentlists()
{
	$styles = vB::getDbAssertor()->assertQuery('vBForum:fetchstyles2');
	foreach ($styles as $style)
	{
		$parentlist = vB_Library::instance('Style')->fetchTemplateParentlist($style['styleid']);
		if ($parentlist != $style['parentlist'])
		{
			vB::getDbAssertor()->assertQuery('vBForum:updatestyleparent', array(
				'parentlist' => $parentlist,
				'styleid' => $style['styleid']
			));
		}
	}

}

// #############################################################################
/**
* Builds all data from the template table into the fields in the style table
*
* @param	boolean $renumber -- no longer used.  Feature removed.
* @param	boolean	If true, will fix styles with no parent style specified
* @param	string	If set, will redirect to specified URL on completion
* @param	boolean	If true, reset the master cache
* @param	boolean	Whether to print status/edit information
*/
function build_all_styles($renumber = 0, $install = 0, $goto = '', $resetcache = false, $printInfo = true)
{
	// -----------------------------------------------------------------------------
	// -----------------------------------------------------------------------------
	// this bit of text is used for upgrade scripts where the phrase system
	// is not available it should NOT be converted into phrases!!!
	$phrases = array(
		'master_style' => 'MASTER STYLE',
		'done' => 'Done',
		'style' => 'Style',
		'styles' => 'Styles',
		'templates' => 'Templates',
		'css' => 'CSS',
		'stylevars' => 'Stylevars',
		'replacement_variables' => 'Replacement Variables',
		'controls' => 'Controls',
		'rebuild_style_information' => 'Rebuild Style Information',
		'updating_style_information_for_each_style' => 'Updating style information for each style',
		'updating_styles_with_no_parents' => 'Updating style sets with no parent information',
		'updated_x_styles' => 'Updated %1$s Styles',
		'no_styles_needed_updating' => 'No Styles Needed Updating',
	);
	$vbphrase = vB_Api::instanceInternal('phrase')->fetch($phrases);
	foreach ($phrases AS $key => $val)
	{
		if (!isset($vbphrase["$key"]))
		{
			$vbphrase["$key"] = $val;
		}
	}
	// -----------------------------------------------------------------------------
	// -----------------------------------------------------------------------------

	$isinstaller = (defined('VB_AREA') AND (VB_AREA == 'Upgrade' OR VB_AREA == 'Install'));
	$doOutput = ($printInfo AND !$isinstaller);

	$form_tags = !empty($goto);
	if ($doOutput)
	{
		echo "<!--<p>&nbsp;</p>-->
		<blockquote>" . ($form_tags ? '<form>' : '') . "<div class=\"tborder\">
		<div class=\"tcat\" style=\"padding:4px\" align=\"center\"><b>" . $vbphrase['rebuild_style_information'] . "</b></div>
		<div class=\"alt1\" style=\"padding:4px\">\n<blockquote>
		";
		vbflush();
	}

	// useful for restoring utterly broken (or pre vb3) styles
	if ($install)
	{
		if ($doOutput)
		{
			echo "<p><b>" . $vbphrase['updating_styles_with_no_parents'] . "</b></p>\n<ul class=\"smallfont\">\n";
			vbflush();
		}

		vB::getDbAssertor()->assertQuery('updt_style_parentlist');
	}

	if ($doOutput)
	{
		// the main bit.
		echo "<p><b>" . $vbphrase['updating_style_information_for_each_style'] . "</b></p>\n";
		vbflush();
	}

	build_template_parentlists();

	$styleactions = array('dostylevars' => 1, 'doreplacements' => 1, 'doposteditor' => 1);
	if (defined('NO_POST_EDITOR_BUILD'))
	{
		$styleactions['doposteditor'] = 0;
	}

	if ($error = build_style(-1, $vbphrase['master_style'], $styleactions, '', '', $resetcache, $printInfo))
	{
		return $error;
	}

	if ($doOutput)
	{
		echo "</blockquote></div>";
		if ($form_tags)
		{
			echo "
			<div class=\"tfoot\" style=\"padding:4px\" align=\"center\">
			<input type=\"button\" class=\"button\" value=\" " . $vbphrase['done'] . " \" onclick=\"window.location='$goto';\" />
			</div>";
		}
		echo "</div>" . ($form_tags ? '</form>' : '') . "</blockquote>
		";
		vbflush();
	}

	vB_Library::instance('Style')->buildStyleDatastore();
}

// #############################################################################
/**
* Displays a style rebuild (build_style) in a nice user-friendly info page
*
* @param	integer	Style ID to rebuild
* @param	string	Title of style
* @param	boolean	Build CSS? (no longer used)
* @param	boolean	Build Stylevars?
* @param	boolean	Build Replacements?
* @param	boolean	Build Post Editor?
*/
function print_rebuild_style($styleid, $title = '', $docss = 1, $dostylevars = 1, $doreplacements = 1, $doposteditor = 1, $printInfo = true)
{
	$vbphrase = vB_Api::instanceInternal('phrase')->fetch(array('master_style', 'rebuild_style_information', 'updating_style_information_for_x', 'done'));
	$styleid = intval($styleid);

	if (empty($title))
	{
		if ($styleid == -1)
		{
			$title = $vbphrase['master_style'];
		}
		else
		{
			$getstyle = vB_Library::instance('Style')->fetchStyleByID($styleid);

			if (!$getstyle)
			{
				return;
			}

			$title = $getstyle['title'];
		}
	}

	if ($printInfo AND (VB_AREA != 'Upgrade') AND (VB_AREA != 'Install'))
	{
		echo "<p>&nbsp;</p>
		<blockquote><form><div class=\"tborder\">
		<div class=\"tcat\" style=\"padding:4px\" align=\"center\"><b>" . $vbphrase['rebuild_style_information'] . "</b></div>
		<div class=\"alt1\" style=\"padding:4px\">\n<blockquote>
		<p><b>" . construct_phrase($vbphrase['updating_style_information_for_x'], $title) . "</b></p>
		<ul class=\"lci\">\n";
		vbflush();
	}

	$actions = array(
		'dostylevars' => $dostylevars,
		'doreplacements' => $doreplacements,
		'doposteditor' => $doposteditor
	);
	build_style($styleid, $title, $actions, false, '', 1, $printInfo);

	if ($printInfo AND (VB_AREA != 'Upgrade') AND (VB_AREA != 'Install'))
	{
		echo "</ul>\n<p><b>" . $vbphrase['done'] . "</b></p>\n</blockquote></div>
		</div></form></blockquote>
		";
		vbflush();
	}

	vB_Library::instance('Style')->buildStyleDatastore();

}

// #############################################################################

/**
 *	Deletes the old style css directory on disk
 *
 *	@param int $styleid
 *	@param string $dir -- the "direction" of the css to delete.  Either 'ltr' or 'rtl' (there are actually two directories per style)
 *	@param bool $contentsonly	-- whether to delete the newly empty directory (if we are deleting the contents to rewrite there is no need)
 */
function delete_style_css_directory($styleid, $dir = 'ltr', $contentsonly = false)
{
	$styledir = vB_Api::instanceInternal('style')->getCssStyleDirectory($styleid, $dir);
	$styledir = $styledir['directory'];

	if (is_dir($styledir))
	{
		$dirhandle = opendir($styledir);

		if ($dirhandle)
		{
			// loop through the files in the style folder
			while (($fname = readdir($dirhandle)) !== false)
			{
				$filepath = $styledir . '/' . $fname;

				// remove just the files inside the directory
				// and this also takes care of the '.' and '..' folders
				if (!is_dir($filepath))
				{
					@unlink($filepath);
				}
			}
			// Close the handle
			closedir($dirhandle);
		}

		// Remove the style directory
		if(!$contentsonly)
		{
			@rmdir($styledir);
		}
	}
}

// #############################################################################
/**
* Attempts to create a new css file for this style
*
* @param	string	CSS filename
* @param	string	CSS contents
*
* @return	boolean	Success
*/
function write_css_file($filename, $contents)
{
	// attempt to write new css file - store in database if unable to write file
	if ($fp = @fopen($filename, 'wb') AND !is_demo_mode())
	{
		fwrite($fp, $contents);
		@fclose($fp);
		return true;
	}
	else
	{
		@fclose($fp);
		return false;
	}
}

/**
 *	Writes style css directory to disk, this includes SVG templates
 *
 *	@param int $styleid
 *	@param string $parentlist -- csv list of ancestors for this style
 *	@param string $dir -- the "direction" of the css to write.  Either 'ltr' or 'rtl' (there are actually two directories per style)
 */
function write_style_css_directory($styleid, $parentlist, $dir = 'ltr')
{
	//verify that we have or can create a style directory
	$styledir = vB_Api::instanceInternal('style')->getCssStyleDirectory($styleid, $dir);
	$styledir = $styledir['directory'];

	//if we have a file that's not a directory or not writable something is wrong.
	if (file_exists($styledir) AND (!is_dir($styledir) OR !is_writable($styledir)))
	{
		return false;
	}

	//clear any old files.
	if (file_exists($styledir))
	{
		delete_style_css_directory($styleid, $dir, true);
	}

	//create the directory -- if it still exists try to continue with the existing dir
	if (!file_exists($styledir))
	{
		if (!mkdir($styledir, 0777, true))
		{
			return false;
		}
	}

	//check for success.
	if (!is_dir($styledir) OR !is_writable($styledir))
	{
		return false;
	}

	// NOTE: SVG templates are processed along with CSS templates.
	// I observed no unwanted behavior by doing this, and if in the
	// future, CSS and SVG templates need to be processed separately
	// we can refactor this at that point.

	//write out the files for this style.
	$parentlistarr = explode(',', $parentlist);
	$set = vB::getDbAssertor()->assertQuery('template_fetch_css_svg_templates', array('styleidlist' => $parentlistarr));

	//collapse the list.
	$css_svg_templates = array();
	foreach ($set as $row)
	{
		$css_svg_templates[] = $row['title'];
	}

	$stylelib = vB_Library::instance('Style');
	$stylelib->switchCssStyle($styleid, $css_svg_templates);

	// Get new css cache bust
	$stylelib->setCssFileDate($styleid);
	$cssfiledate = $stylelib->getCssFileDate($styleid);

	// Keep pseudo stylevars in sync with the css.php and sprite.php handling
	set_stylevar_ltr(($dir == 'ltr'));
	set_stylevar_meta($styleid);

	$base = get_base_url_for_css();
 	if ($base === false)
	{
		return false;
	}

	$templates = array();
	$templates_not_in_rollups = array();
	foreach ($css_svg_templates AS $title)
	{
		$text = vB_Template::create($title)->render(true);
		$text = css_file_url_fix($base, $text);
		$text = vB_String::getCssMinifiedText($text);

		$templates[$title] = $text;
		$templates_not_in_rollups[$title] = true;
	}

	static $vbdefaultcss = array(), $cssfiles = array();

	if (empty($vbdefaultcss))
	{
		$cssfilelist = vB_Api::instanceInternal('product')->loadProductCssRollups();

		if (empty($cssfilelist['vbulletin']))
		{
			$vbphrase = vB_Api::instanceInternal('phrase')->fetch(array('could_not_open_x'));
			echo construct_phrase($vbphrase['could_not_open_x'], DIR . '/includes/xml/cssrollup_vbulletin.xml');
			exit;
		}

		$data = $cssfilelist['vbulletin'];
		unset($cssfilelist['vbulletin']);

		if (!is_array($data['rollup'][0]))
		{
			$data['rollup'] = array($data['rollup']);
		}

		foreach ($data['rollup'] AS $file)
		{
			foreach ($file['template'] AS $name)
			{
				$vbdefaultcss["$file[name]"] = $file['template'];
			}
		}

		foreach ($cssfilelist AS $css_file => $data)
		{
			$products = vB::getDatastore()->getValue('products');
			if ($data['product'] AND empty($products["$data[product]"]))
			{
				// attached to a specific product and that product isn't enabled
				continue;
			}

			if (!is_array($data['rollup'][0]))
			{
				$data['rollup'] = array($data['rollup']);
			}

			$cssfiles[$css_file]['css'] = $data['rollup'];
		}
	}


	foreach ($cssfiles AS $css_file => $files)
	{
		if (is_array($files['css']))
		{
			foreach ($files['css'] AS $file)
			{
				$result = process_css_rollup_file($file['name'], $file['template'], $templates,
					$styleid, $styledir, $templates_not_in_rollups, $vbdefaultcss);
				if ($result === false)
				{
					return false;
				}
			}
		}
	}

	foreach ($vbdefaultcss AS $xmlfile => $files)
	{
		$result = process_css_rollup_file($xmlfile, $files, $templates, $styleid, $styledir, $templates_not_in_rollups);
		if ($result === false)
		{
			return false;
		}
	}

	foreach($templates_not_in_rollups AS $title => $dummy)
	{
		if (!write_css_file("$styledir/$cssfiledate-$title", $templates[$title]))
		{
			return false;
		}
	}

	return true;
}


//I'd call this a hack but there probably isn't a cleaner way to do this.
//The css is published to a different directory than the css.php file
//which means that relative urls that works for css.php won't work for the
//published directory.  Unfortunately urls from the webroot don't work
//because the forum often isn't located at the webroot and we can only
//specify urls from the forum root.  And css doens't provide any way
//of setting a base url like html does.  So we are left to "fixing"
//any relative urls in the published css.
//
//We leave alone any urls starting with '/', 'http', 'https:', 'data:', and '#'
//URLs starting with # are for SVG sprite filter elements (not entirely clear on
//what these are for but *probably* internal links in the SVG file that need to
//stay relative to that file and would be broken by adding the base url)
//there are other valid urls, but nothing that people should be
//using in our css files.
function css_file_url_fix($base, $text)
{
	$re = '#(url\(\s*["\']?)([^)]*\))#';
	$callback = function($matches) use ($base)
	{
		//we aren't really guarenteed that contents is the full contents of the
		//url(...) block.  There are several was a closing paren can sneak in so that
		//we match that but the css parser will skip it.  For instance
		//url("http://www.google.com?x=()") is a valid url and won't break the css
		//however all we really care about is that we get the prefix portion of the
		//url, that we stop before we hit the next url, and that we preserve anything
		//that we don't match -- we just need to be careful to do that last bit instead
		//of assuming we're replacing everything.
		list($all, $prefix, $contents) = $matches;

		//if we have a prefix that suggests some kind of absolute url, leave it alone
		if(preg_match('#^(?:/|http:|https:|data:|\#)#', $contents))
		{
			return $all;
		}
		else
		{
			//otherwise insert the base
			return $prefix . $base . $contents;
		}
	};

	$text = preg_replace_callback($re, $callback, $text);
	return $text;
}



/**
 *	Gets the base url for images in the css files
 *
 *	This will be the site root unless there is a CDN configured
 *	the image will all be specified in the css to the url the css.php file is located.
 *	When writing the css to disk we make the urls absolute because the paths are different
 *	from the static css files.
 */
function get_base_url_for_css()
{
	/*
		We need the frontend base url.  This should be always available but if not
		try to check the backend config.  Not sure this is needed but the old code does it.
		By default this isnt set, but the site administrator can set it.
		If all this fails, we give up and return false
	*/

	$base = vB::getDatastore()->getOption('cdnurl');
	if (!$base)
	{
		$baseurl = '';
		if ($frontendurl = vB::getDatastore()->getOption('frontendurl'))
		{
			$baseurl = $frontendurl;
		}
		else
		{
			$config = vB::getConfig();
			$baseurl = $config['Misc']['baseurl'];
		}

		$base = $baseurl;
	}

	if (substr($base, -1, 1) != '/')
	{
		$base .= '/';
	}

	return $base;
}


function process_css_rollup_file(
	$file,
	$templatelist,
	$templates,
	$styleid,
	$styledir,
	&$templates_not_in_rollups,
	&$vbdefaultcss = array()
)
{
	if (!is_array($templatelist))
	{
		$templatelist = array($templatelist);
	}

	if ($vbdefaultcss AND $vbdefaultcss["$file"])
	{
		// Add these templates to the main file rollup
		$vbdefaultcss["$file"] = array_unique(array_merge($vbdefaultcss["$file"], $templatelist));
		return true;
	}

	$count = 0;
	$text = "";
	foreach ($templatelist AS $name)
	{
		unset($templates_not_in_rollups[$name]);
		$template = $templates[$name];
		if ($count > 0)
		{
			$text .= "\r\n\r\n";
			$template = preg_replace("#@charset [^;]*;#i", "", $template);
		}
		$text .= $template;
		$count++;
	}

	$stylelib = vB_Library::instance('Style');
	$cssfiledate = $stylelib->getCssFileDate($styleid);

	if (!write_css_file("$styledir/$cssfiledate-$file", $text))
	{
		return false;
	}

	return true;
}

// #############################################################################
/**
* Converts all data from the template table for a style into the style table
*
* @param	integer	Style ID
* @param	string	Title of style
* @param	array	Array of actions set to true/false: dostylevars/doreplacements/doposteditor
* @param	string	List of parent styles
* @param	string	Indent for HTML printing
* @param	boolean	Reset the master cache
* @param	boolean	Whether to print status/edit information
*/
function build_style($styleid, $title = '', $actions = [], $parentlist = '', $indent = '', $resetcache = false, $printInfo = true)
{
	//not sure if this is required.
	require_once(DIR . '/includes/adminfunctions.php');

	$db = vB::getDbAssertor();
	$datastore = vB::getDatastore();
	$styleLib = vB_Library::instance('Style');

	$isinstaller = (defined('VB_AREA') AND (VB_AREA == 'Upgrade' OR VB_AREA == 'Install'));
	$doOutput = ($printInfo AND !$isinstaller);

	//we only use the phrases if we are doing some output.  No need to load them if we aren't going to use them.
	//note that they are cached so we aren't querying the DB for each style
	if($doOutput)
	{
		$vbphrase = vB_Api::instanceInternal('phrase')->fetch(['templates', 'stylevars', 'replacement_variables', 'controls', 'done']);
	}

	//don't propagate any local changes to actions to child rebuilds.
	$originalactions = $actions;
	if ($styleid != -1)
	{
		$usecssfiles = $styleLib->useCssFiles($styleid);

		//this is some *old* code.  I think it's due to some fields that writing css files
		//relies on not getting set, but it's been copied, tweaked, and mangled since cssasfiles
		//referred to the vB3 css and not the css template sheets so it's not 100% if it's needed
		//any longer.
		if (($actions['doreplacements'] OR $actions['dostylevars']) AND $usecssfiles)
		{
			$actions['doreplacements'] = true;
		}

		// VBV-16291 certain actions, like write_css_file(), relies on in-memory cached items that would normally be cleared
		// if going through the styleLIB's buildStyle(). To avoid stale data issues in upgrade, we manually clear it here.
		$styleLib->internalCacheClear($styleid);
		if ($doOutput)
		{
			// echo the title and start the listings
			echo "$indent<li><b>$title</b> ... <span class=\"smallfont\">";
			vbflush();
		}

		// build the templateid cache
		if (!$parentlist)
		{
			$parentlist = $styleLib->fetchTemplateParentlist($styleid);
		}

		$templatelist = $styleLib->buildTemplateIdCache($styleid, 1, $parentlist);

		$styleupdate = [];
		$styleupdate['templatelist'] = $templatelist;

		if ($doOutput)
		{
			echo "($vbphrase[templates]) ";
			vbflush();
		}

		// cache special templates
		if ($actions['doreplacements'] OR $actions['doposteditor'])
		{
			// get special templates for this style
			$template_cache = [
				'replacement' => [],
				'template' => [],
			];
			$templateids = unserialize($templatelist);
			$specials = vB_Api::instanceInternal('template')->fetchSpecialTemplates();

			if ($templateids)
			{
				$templates = $db->assertQuery('vBForum:fetchtemplatewithspecial', array(
					'templateids' => $templateids,
					'specialtemplates' => $specials
				));

				foreach ($templates as $template)
				{
					$template_cache["$template[templatetype]"]["$template[title]"] = $template;
				}
			}
		}

		// style vars
		if ($actions['dostylevars'])
		{
			// new stylevars
			static $master_stylevar_cache = null;
			static $resetcachedone = false;
			if ($resetcache AND !$resetcachedone)
			{
				$resetcachedone = true;
				$master_stylevar_cache = null;
			}

			if ($master_stylevar_cache === null)
			{
				$master_stylevar_cache = [];
				$master_stylevars = $db->assertQuery('vBForum:getDefaultStyleVars');

				foreach ($master_stylevars AS $master_stylevar)
				{
					$tmp = '';
					if(isset($master_stylevar['value']))
					{
						$tmp = unserialize($master_stylevar['value']);
					}

					if (!is_array($tmp))
					{
						$tmp = ['value' => $tmp];
					}
					$master_stylevar_cache[$master_stylevar['stylevarid']] = $tmp;
					$master_stylevar_cache[$master_stylevar['stylevarid']]['datatype'] = $master_stylevar['datatype'];
				}
			}

			$newstylevars = $master_stylevar_cache;

			if (substr(trim($parentlist), 0, -3) != '')
			{
				$data = array(
					'stylelist' => explode(',', substr(trim($parentlist), 0, -3)),
					'parentlist' => $parentlist,
				);
				$new_stylevars = $db->getRows('vBForum:getStylesFromList', $data);

				foreach ($new_stylevars AS $new_stylevar)
				{
					$new_stylevarid = $new_stylevar['stylevarid'];

					ob_start();
					$newstylevars[$new_stylevarid] = unserialize($new_stylevar['value']);
					if (ob_get_clean() OR !is_array($newstylevars[$new_stylevarid]))
					{
						continue;
					}

					//this most likely means we have an orphaned stylevar -- which is not good but it isn't going
					//to be helped by accessing an invalid array index.
					if(isset($master_stylevar_cache[$new_stylevarid]))
					{
						$newstylevars[$new_stylevarid]['datatype'] = $master_stylevar_cache[$new_stylevarid]['datatype'];
					}
				}
			}

			$styleupdate['newstylevars'] = serialize($newstylevars);

			if ($doOutput)
			{
				echo "($vbphrase[stylevars]) ";
				vbflush();
			}
		}

		// replacements
		if ($actions['doreplacements'])
		{
			// rebuild the replacements field for this style
			$replacements = [];
			if (is_array($template_cache['replacement']))
			{
				foreach($template_cache['replacement'] AS $template)
				{
					// set the key to be a case-insentitive preg find string
					$replacementkey = '#' . preg_quote($template['title'], '#') . '#si';

					$replacements["$replacementkey"] = $template['template'];
				}
				$styleupdate['replacements'] = serialize($replacements) ;
			}
			else
			{
				$styleupdate['replacements'] = "''";
			}

			if ($doOutput)
			{
				echo "($vbphrase[replacement_variables]) ";
				vbflush();
			}
		}

		// post editor styles
		if ($actions['doposteditor'] AND $template_cache['template'])
		{
			$editorstyles = [];
			if (!empty($template_cache['template']))
			{
				foreach ($template_cache['template'] AS $template)
				{
					if (substr($template['title'], 0, 13) == 'editor_styles')
					{
						$title = 'pi' . substr($template['title'], 13);
						$item = fetch_posteditor_styles($template['template']);
						$editorstyles["$title"] = [$item['background'], $item['color'], $item['padding'], $item['border']];
					}
				}
			}
			if  ($doOutput)
			{
				echo "($vbphrase[controls]) ";
				vbflush();
			}
		}

		// do the style update query
		if (!empty($styleupdate))
		{
			$styleupdate['styleid'] = $styleid;
			$styleupdate[vB_dB_Query::TYPE_KEY] = vB_dB_Query::QUERY_UPDATE;
			$db->assertQuery('vBForum:style', $styleupdate);
		}

		//write out the new css -- do this *after* we update the style record
		if ($usecssfiles)
		{
			//restore the current style settings to avoid affecting downstream display.
			$originaldirection = vB_Template_Runtime::fetchStyleVar('textdirection');
			$originalstyleid = vB::getCurrentSession()->get('styleid');

			foreach(['ltr', 'rtl'] AS $direction)
			{
				if (!write_style_css_directory($styleid, $parentlist, $direction))
				{
					$error = fetch_error("rebuild_failed_to_write_css");
					if ($doOutput)
					{
						echo $error;
					}
					else
					{
						return $error;
					}
				}
			}

			set_stylevar_ltr($originaldirection == 'ltr');
			set_stylevar_meta($originalstyleid);
		}

		// finish off the listings
		if ($doOutput)
		{
			echo "</span><b>" . $vbphrase['done'] . "</b>.<br />&nbsp;</li>\n";
			vbflush();
		}
	}

	$childsets = $db->getRows('style', ['parentid' => $styleid]);
	if (count($childsets))
	{
		if ($doOutput)
		{
			echo "$indent<ul class=\"ldi\">\n";
		}

		foreach ($childsets as $childset)
		{
			if ($error = build_style($childset['styleid'], $childset['title'], $originalactions, $childset['parentlist'], $indent . "\t", $resetcache, $printInfo))
			{
				return $error;
			}
		}

		if ($doOutput)
		{
			echo "$indent</ul>\n";
		}
	}

	//We want to force a fastDS rebuild, but we can't just call rebuild. There may be dual web servers,
	// and calling rebuild only rebuilds one of them.
	$options = $datastore->getValue('miscoptions');
	$options['tmtdate'] = vB::getRequest()->getTimeNow();
	$datastore->build('miscoptions', serialize($options), 1);
}

// #############################################################################
//the print_style functions are closely tied to the template.php style manager and are
//unlikely to be useful for anything else.  In particular the logic replies on
//all of the styles being present (potentially aside from the read protected themes)
//along with some javascript set up that only happens there.  This isn't something that
//needs fixing or particularly can be fixed but it's something to be aware of.

/**
* Prints out a style editor block
*
* @param	integer	Style ID
* @param	array	Style info array
*/
function print_style($masterset, $style, $colspan)
{
	global $vbulletin;

	$styleid = $style['styleid'];

	//we really shouldn't be looking up query string/post info inside of functions
	//but at least we can pull it out to the top of the first function instead of
	//embedding it
	$titlesonly = $vbulletin->GPC['titlesonly'];
	$expandset = $vbulletin->GPC['expandset'];
	$group = $vbulletin->GPC['group'];
	$searchstring = $vbulletin->GPC['searchstring'];

	//this is *probably* always set, based on where the function gets called,
	//but the original code had a guard on it and so we'll make sure.
 	$templateid = (!empty($vbulletin->GPC['templateid']) ? $vbulletin->GPC['templateid'] : null);

	$vbphrase = vB_Api::instanceInternal('phrase')->fetch([
		'add_child_style', 'add_new_template', 'all_template_groups', 'allow_user_selection',
		'collapse_all_template_groups', 'collapse_template_group', 'collapse_templates',
		'collapse_x', 'common_templates', 'controls', 'custom_templates', 'customize_gstyle',
		'delete_style', 'edit_display_order', 'download', 'edit', 'edit_fonts_colors_etc', 'edit_settings_gstyle',
		'edit_style_options', 'edit_templates', 'expand_all_template_groups', 'expand_template_group',
		'expand_templates', 'expand_x', 'go', 'replacement_variables', 'revert_all_stylevars',
		'revert_all_templates', 'revert_gcpglobal', 'stylevareditor', 'template_is_customized_in_this_style',
		'template_is_inherited_from_a_parent_style', 'template_is_unchanged_from_the_default_style',
		'template_options', 'view_original', 'view_your_forum_using_this_style', 'x_templates', 'choose_action', 'color_key'
	]);

	$vb5_config = vB::getConfig();
	//in debug mode we show the MASTER style as a the root so we need an extra depth level
	if($vb5_config['Misc']['debug'] AND $styleid != -1)
	{
		$style['depth']++;
	}


	$showstyle = ($expandset == 'all' OR $expandset == $styleid);

	// show the header row
	print_style_header_row($vbphrase, $style, $group, $showstyle);
	if ($showstyle)
	{
		//Need to figure out how to ensure that the templatelist is properly passed so
		//we don't need this silliness.  But for now pulling it to the top level function
		//for visibility.
		//master style doesn't really exist and therefore can't be loaded.
		if (empty($style['templatelist']))
		{
	 		$style = vB_Library::instance('Style')->fetchStyleById($styleid);
		}

		/*
			If $style was passed into this function, templatelist might still be a serialized string.
			Note, I only ever ran into this once, and it never happened again, so it might've been
			some old cached data somewhere (cached before the removal of templatelist from the stylecache,
			maybe?)
		 */
		if (is_string($style['templatelist']))
		{
			$style['templatelist'] = unserialize($style['templatelist']);
		}

		$groups = vB_Library::instance('template')->getTemplateGroupPhrases();
		$template_groups = vB_Api::instanceInternal('phrase')->renderPhrases($groups);
		$template_groups = $template_groups['phrases'];

		$templates = print_style_get_templates($vbulletin->db, $style, $masterset, $template_groups, $searchstring, $titlesonly);
	 	if($templates)
		{
			echo '<tr><td colspan="' . $colspan . '" style="padding: 0px;">' . "\n";
			print_style_body($vbphrase, $templates, $template_groups, $style, $group, $templateid, $expandset);
			echo '</td></tr>' . "\n";
		}
	}
}


// Function to break up print_style into something readable.  Should be considered "private" to this file
function print_style_header_row($vbphrase, $style, $group, $showstyle)
{
	$styleid = $style['styleid'];

	$styleLib = vB_Library::instance('style');
	$showReadonlyMarking = !$styleLib->checkStyleReadProtection($styleid, $style, true);

	$cells = [];

	//cell for style title block.
	$title = $style['title'];
	if ($showReadonlyMarking)
	{
		$title = $style['title'] . ' <span class="acp-style-readonly-mark"></span>';
	}

	$label = '&nbsp; ' . construct_depth_mark($style['depth'], '- - ');
	if($styleid != -1)
	{
		$selectname = 'userselect[' . $styleid . ']';
		if(!$showReadonlyMarking)
		{
			$attributes = [
				'value' => 1,
				'class' => 'bginput js-template-userselect',
				'data-styleid' => $style['styleid'],
				'data-parentid' => $style['writableparentid'],
			];
			if($style['userselect'])
			{
				$attributes['checked'] = 'checked';
			}

			$userselect = construct_input('checkbox', $selectname, $attributes, false);
			$label = '<label title="' . $vbphrase['allow_user_selection'] . '">' . $label . $userselect . '</label>';
		}
		else
		{
			//We probably don't want to ever have the hidden themes user selectable.  But turning
			//the flag off as a random side effect also seems like a poor choice.  The quick save
			//assumes that all of the styles have a value for the select input and, because checkboxes
			//don't show in forms if they are unchecked there isn't a good way to inspect the form
			//to determine if the checkbox is present or not.
			//This preserves the value if we don't allow it to be set.
			construct_hidden_code($selectname, $style['userselect']);
		}
	}

	//we use the anchor to scroll the list when returning to the open style.  This avoids the user having to constantly find
	//the style they're working on in a long list.
	$anchor = '<span id="styleheader' . $styleid . '" ></span>';
	$forumhome_url = vB5_Route::buildUrl('home|fullurl', [], ['styleid' => $styleid]);
	$cells[] = $anchor . $label . '<a href="' . $forumhome_url . '" target="_blank" title="' . $vbphrase['view_your_forum_using_this_style'] . '">' . $title . '</a>';

	//cell for tools.
	$displayorder = '';
	if($styleid != -1)
	{
		$attributes = [
			'value' => $style['displayorder'],
			'class' => 'bginput display-order',
			'title' => $vbphrase['edit_display_order'],
		];
		$displayorder = construct_input('text', 'displayorder[' . $styleid . ']', $attributes);
	}

	$menuname = ($styleid != -1 ? 'styleEdit_' . $styleid : 'styleEdit_m');
	$selecttext = print_style_get_command_select($vbphrase, $menuname, $style);
	$data = ['menu' => $menuname];
	$goButton = construct_event_button($vbphrase['go'], 'js-button-template-go', $data);

	if($showstyle)
	{
		$code = COLLAPSECODE;
		$tooltip = $vbphrase['collapse_templates'];
	}
	else
	{
		$code = EXPANDCODE;
		$tooltip = $vbphrase['expand_templates'];
	}

	$data = [
		'group' => $group,
		'styleid' => ($showstyle ? '' : $styleid),
	];
	$expandCollapse = construct_event_button($code, 'js-button-template-expand', $data, [], ['title' => $tooltip]);


	$cells[] = $displayorder;
	$cells[] = $selecttext . $goButton;
	$cells[] = $expandCollapse;

	print_cells_row2($cells, 'nowrap ' . fetch_row_bgclass());
}

function print_style_get_command_select($vbphrase, $menuname, $style)
{
	$styleid = $style['styleid'];

	//this is probably not useful any longer.  If this returns read only the style got filtered
	//out of the list before we got here.
	$styleLib = vB_Library::instance('style');
	$styleIsReadonly = !$styleLib->checkStyleReadProtection($styleid, $style);

	//canadminstyles no and canadmintemplates yes isn't really a sensible combination as a result some of the behavior in this
	//instance is inconsistant.  I'm not sure if you can even get to this function without it an if so if you should.
	//We need to make sure that only canadminstyles and both are well handled.
	$userContext = vB::getUserContext();
	$canadminstyles = $userContext->hasAdminPermission('canadminstyles');
	$canadmintemplates = $userContext->hasAdminPermission('canadmintemplates');

	//this should be all possible option groups.  We'll control via permissions later on.
	//Only groups with options will be displayed.  Order in this array controls display order.
	//We assume that everything except the default "choose" option is in an option group
	//Note that if display_order is not unique then order is undefined (and can vary based on inconsequential changes)
	//We do it this way so that we don't have to consider whether to display an option in the order that we
	//intend to display it -- thus allowing us to avoid massive duplication of logic.
	$optgroups = [
		'template_options' => [],
		'edit_fonts_colors_etc' => [],
		'edit_style_options' => [],
	];

	//these options for for the more detailed template edit options.
	if ($canadmintemplates)
	{
		if(!$styleIsReadonly)
		{
			//these do not apply to the master style
			if($styleid != -1)
			{
				$optgroups['template_options'][30] = ['phrase' => 'revert_all_templates', 'action' => 'template_revertall'];
			}
		}

		$optgroups['edit_style_options'][30] = ['phrase' => 'download', 'action' => 'template_download'];
	}

	//we now allow this regardless of which permission the admin has
	if(!$styleIsReadonly)
	{
		$optgroups['template_options'][10] = ['phrase' => 'edit_templates', 'action' => 'template_templates'];
		$optgroups['template_options'][20] = ['phrase' => 'add_new_template', 'action' => 'template_addtemplate'];
	}

	if ($canadminstyles)
	{
		if(!$styleIsReadonly)
		{
			$optgroups['edit_fonts_colors_etc'][20] = ['phrase' => 'stylevareditor', 'action' => 'stylevar'];

			if($styleid != -1)
			{
				$optgroups['edit_fonts_colors_etc'][30] = ['phrase' => 'revert_all_stylevars', 'action' => 'stylevar_revertall'];
			}

			$optgroups['edit_fonts_colors_etc'][40] = ['phrase' => 'replacement_variables', 'action' => 'css_replacements'];
		}

		if($styleid != -1)
		{
			$optgroups['edit_style_options'][10] = ['phrase' => 'edit_settings_gstyle', 'action' => 'template_editstyle'];
			$optgroups['edit_style_options'][40] = ['phrase' => 'delete_style', 'action' => 'template_delete', 'class' => 'col-c'];
		}

		$optgroups['edit_style_options'][20] = ['phrase' => 'add_child_style', 'action' => 'template_addstyle'];
	}


	$optgrouptext = '';
	foreach($optgroups AS $groupphrase => $options)
	{
		if($options)
		{
			ksort($options);
			$optgrouptext .= '<optgroup label="' . $vbphrase[$groupphrase] . '">' . "\n";
			foreach($options AS $option)
			{
				$class = '';
				if(isset($option['class']))
				{
					$class = ' class="' . $option['class'] . '"';
				}

				$optgrouptext .= '<option value="' . $option['action'] . '"' . $class . '>' . $vbphrase[$option['phrase']] . "</option>\n";
			}
			$optgrouptext .= "</optgroup>\n";
		}
	}

	$select = "<select name=\"$menuname\" data-styleid=\"$styleid\" class=\"js-template-menu bginput picklist\">
		<option selected=\"selected\">" . $vbphrase['choose_action'] . "</option>
		$optgrouptext
		</select>"; //a newline here causes a display change.

	return $select;
}

// Function to break up print_style into something readable.  Should be considered "private" to this file
function print_style_get_templates($db, $style, $masterset, $template_groups, $searchstring, $titlesonly)
{
	$searchconds = [];

	if (!empty($searchstring))
	{
		$containsSearch = "LIKE('%" . $db->escape_string_like($searchstring) . "%')";
		if ($titlesonly)
		{
			$searchconds[] = "t1.title $containsSearch";
		}
		else
		{
			$searchconds[] = "( t1.title $containsSearch OR template_un $containsSearch ) ";
		}
	}

	// not sure if this if is necesary any more.  The template list should always be set
	// at this point and if it isn't things don't work properly.  However I need to stop
	// fixing things at some point and wrap this up.
	if (!empty($style['templatelist']) AND is_array($style['templatelist']))
	{
		$templateids = implode(',' , $style['templatelist']);
		if (!empty($templateids))
		{
			$searchconds[] = "templateid IN($templateids)";
		}
	}

	$specials = vB_Api::instanceInternal('template')->fetchSpecialTemplates();

	$sql = "
		SELECT templateid, IF(((t1.title LIKE '%.css') AND (t1.title NOT like 'css_%')),
			CONCAT('csslegacy_', t1.title), title) AS title, styleid, templatetype, dateline, username
		FROM " . TABLE_PREFIX . "template AS t1
		WHERE
			templatetype IN('template', 'replacement') AND " . implode(' AND ', $searchconds) . "
		AND title NOT IN('" . implode("', '", $specials) . "')
		ORDER BY title
	";

	$templates = $db->query_read($sql);

	// just exit if no templates found
	$numtemplates = $db->num_rows($templates);
	if ($numtemplates == 0)
	{
		return false;
	}

	$result = [
		'replacements' => [],
		'customtemplates' => [],
		'maintemplates' => [],
	];

	while ($template = $db->fetch_array($templates))
	{
		$templateid = $template['templateid'];
		if ($template['templatetype'] == 'replacement')
		{
			$result['replacements'][$templateid] = $template;
		}
		else
		{
			// don't show special templates
			if (in_array($template['title'], $specials))
			{
				continue;
			}

			$title = $template['title'];

			$groupname = explode('_', $title, 2);
			$groupname = $groupname[0];

			if ($template['styleid'] != -1 AND !isset($masterset[$title]) AND !isset($template_groups[$groupname]))
			{
				$result['customtemplates'][$templateid] = $template;
			}
			else
			{
				$result['maintemplates'][$templateid] = $template;
			}
		}
	}

	return $result;
}

// Function to break up print_style into something readable.  Should be considered "private" to this file
function print_style_body($vbphrase, $templates, $template_groups, $style, $group, $selectedtemplateid, $expandset)
{
	$userContext = vB::getUserContext();
	$canadmintemplates = $userContext->hasAdminPermission('canadmintemplates');

	$directionLeft = vB_Template_Runtime::fetchStyleVar('left');
	$styleid = $style['styleid'];

	//it's really not clear *why* we need to do this.  But in some instances we need an id of 0
	//instead of -1 for the master style.  Need to figure that out, but for now we'll keep the
	//previous behavior.
	//It seems like the JS variables use 0 as the key for the master style.
	$THISstyleid = $styleid;
	if($styleid == -1)
	{
		$THISstyleid = 0;
	}

	echo '
		<!-- start template list for style "' . $style['styleid'] . '" -->
		<table cellpadding="0" cellspacing="10" border="0" align="center">
			<tr valign="top">
				<td>
					<select data-styleid = "' . $THISstyleid . '" name="tl' . $THISstyleid . '" class="darkbg js-templatelist" size="' . TEMPLATE_EDITOR_ROWS . '" style="width:450px">
					<option class="templategroup" value="">- - ' . construct_phrase($vbphrase['x_templates'], $style['title']) . ' - -</option>
	';

	// custom templates
	if (!empty($templates['customtemplates']))
	{
		echo "<optgroup label=\"\">\n";
		echo "\t<option class=\"templategroup\" value=\"\">" . $vbphrase['custom_templates'] . "</option>\n";

		foreach($templates['customtemplates'] AS $template)
		{
			echo construct_template_option($template, $selectedtemplateid, $styleid);
			vbflush();
		}

		echo '</optgroup>';
	}

	// main templates
	if ($canadmintemplates AND !empty($templates['maintemplates']))
	{
		$lastgroup = '';
		$echo_ul = 0;

		foreach($templates['maintemplates'] AS $template)
		{
			$showtemplate = 1;
			if (!empty($lastgroup) AND isTemplateInGroup($template['title'], $lastgroup))
			{
				if ($group == 'all' OR $group == $lastgroup)
				{
					echo construct_template_option($template, $selectedtemplateid, $styleid);
					vbflush();
				}
			}
			else
			{
				foreach($template_groups AS $thisgroup => $display)
				{
					if ($lastgroup != $thisgroup AND $echo_ul == 1)
					{
						echo "</optgroup>";
						$echo_ul = 0;
					}

					if (isTemplateInGroup($template['title'], $thisgroup))
					{
						$lastgroup = $thisgroup;
						if ($group == 'all' OR $group == $lastgroup)
						{
							//don't select a group if we are selecting a template
							$selected = '';
							if($group == $thisgroup AND !$selectedtemplateid)
							{
								$selected = ' selected="selected"';
							}

							echo "<optgroup label=\"\">\n";
							echo "\t<option class=\"templategroup\" value=\"[]\"" . $selected . ">" .
								construct_phrase($vbphrase['x_templates'], $display) . " &laquo;</option>\n";
							$echo_ul = 1;
						}
						else
						{
							echo "\t<option class=\"templategroup\" value=\"[$thisgroup]\">" . construct_phrase($vbphrase['x_templates'], $display) . " &raquo;</option>\n";
							$showtemplate = 0;
						}
						break;
					}
				} // end foreach($template_groups

				if ($showtemplate)
				{
					echo construct_template_option($template, $selectedtemplateid, $styleid);
					vbflush();
				}
			} // end if template string same AS last
		}
	}

	$data =  ['group' => 'all','styleid' => $expandset];
	$expandbutton = construct_event_button(EXPANDCODE, 'js-button-template-expand', $data, [], ['title' => $vbphrase['expand_all_template_groups']]);

	$data =  ['group' => '','styleid' => $expandset];
	$collapsebutton = construct_event_button(COLLAPSECODE, 'js-button-template-expand', $data, [], ['title' => $vbphrase['collapse_all_template_groups']]);

	$data =  ['styleid' => $THISstyleid, 'request' => ''];
	$customizebutton = construct_event_button($vbphrase['customize_gstyle'], 'js-button-template-action', $data, [], ['id' => 'cust' . $THISstyleid]);

	$title = trim(construct_phrase($vbphrase['expand_x'], '')) . '/' . trim(construct_phrase($vbphrase['collapse_x'], ''));
	$menuexpandbutton = construct_event_button($title, 'js-button-template-action', $data, [], ['id' => 'expa' . $THISstyleid]);

	$editbutton = construct_event_button($vbphrase['edit'], 'js-button-template-action', $data, [], ['id' => 'edit' . $THISstyleid]);

	$data['request'] = 'vieworiginal';
	$viewbutton = construct_event_button($vbphrase['view_original'], 'js-button-template-action', $data, [], ['id' => 'orig' . $THISstyleid]);

	$data['request'] = 'killtemplate';
	$killbutton = construct_event_button($vbphrase['revert_gcpglobal'], 'js-button-template-action', $data, [], ['id' => 'kill' . $THISstyleid]);

	echo '
		</select>
	</td>';

	echo "
	<td width=\"100%\" align=\"center\" valign=\"top\">
	<table cellpadding=\"4\" cellspacing=\"1\" border=\"0\" class=\"tborder\" width=\"300\">
	<tr align=\"center\">
		<td class=\"tcat\"><b>$vbphrase[controls]</b></td>
	</tr>
	<tr>
		<td class=\"alt2\" align=\"center\" style=\"font: 11px tahoma, verdana, arial, helvetica, sans-serif\"><div style=\"margin-bottom: 4px;\">\n" .
			$customizebutton . "\n" .
			$menuexpandbutton .
			"</div>\n" .
			$editbutton . "\n" .
			$viewbutton . "\n" .
			$killbutton . "\n" .
			"<div class=\"darkbg\" style=\"margin: 4px; padding: 4px; border: 2px inset; text-align: " . $directionLeft . "\" id=\"helparea$THISstyleid\">
				" . construct_phrase($vbphrase['x_templates'], '<b>' . $style['title'] . '</b>') . "
			</div>\n" .
			$expandbutton . "\n",
			'<b>' . $vbphrase['all_template_groups'] . "</b>\n" .
			$collapsebutton . "\n" .
		"</td>
	</tr>
	</table>
	<br />
	<table cellpadding=\"4\" cellspacing=\"1\" border=\"0\" class=\"tborder\" width=\"300\">
	<tr align=\"center\">
		<td class=\"tcat\"><b>$vbphrase[color_key]</b></td>
	</tr>
	<tr>
		<td class=\"alt2\">
		<div class=\"darkbg\" style=\"margin: 4px; padding: 4px; border: 2px inset; text-align: " . $directionLeft . "\">
		<span class=\"col-g\">" . $vbphrase['template_is_unchanged_from_the_default_style'] . "</span><br />
		<span class=\"col-i\">" . $vbphrase['template_is_inherited_from_a_parent_style'] . "</span><br />
		<span class=\"col-c\">" . $vbphrase['template_is_customized_in_this_style'] . "</span>
		</div>
		</td>
	</tr>
	</table>
	";

	echo "\n</td>\n</tr>\n</table>\n";
	echo "<!-- end template list for style '$style[styleid]' -->\n\n";
}

/**
 * Tests if the given template is part of the template "group"
 *
 * @param	string	Template name
 * @param	string	Template "group" name
 *
 * @return	bool	True if the given template is part of the group, false otherwise
 */
function isTemplateInGroup($templatename, $groupname)
{
	return (strpos(strtolower(" $templatename"), $groupname) == 1);
}

// #############################################################################
/**
* Constructs a single template item for the style editor form
*
* @param	array	Template info array
* @param	integer	Style ID of style being shown
* @param	boolean	No longer used
* @param	boolean	HTMLise template titles?
*
* @return	string	Template <option>
*/
function construct_template_option($template, $selectedtemplateid, $styleid)
{
	$selected = '';
	if ($selectedtemplateid == $template['templateid'])
	{
		$selected = ' selected="selected"';
	}

	//deal with the title.  The csslegacy thing is a hack we can probably remove -- but that will
	//require a bit of work to make sure we don't break anything.
	$title = $template['title'];
	$title = preg_replace('#^csslegacy_#i', '', $title);
	$title = htmlspecialchars_uni($title);

	$i = $template['username'] . ';' . $template['dateline'];

	$tsid = '';

	if ($styleid == -1)
	{
		$class = '';
		$value = $template['templateid'];
	}
	else
	{
		switch ($template['styleid'])
		{
			// template is inherited from the master set
			case 0:
			case -1:
			{
				$class = 'col-g';
				$value = '~';
				break;
			}

			// template is customized for this specific style
			case $styleid:
			{
				$class = 'col-c';
				$value = $template['templateid'];
				break;
			}

			// template is customized in a parent style - (inherited)
			default:
			{
				$class = 'col-i';
				$value = '[' . $template['templateid'] . ']';
				$tsid = $template['styleid'];
				break;
			}
		}
	}

	$option = "\t" . '<option class="template-option ' . $class . '" value="' . $value . '" ';
	if($tsid)
	{
		$option .= 'tsid="' . $tsid . '" ';
	}
	$option .= 'i="' . $i . '"' . $selected . '>' . $title . "</option>\n";

	return $option;
}

// #############################################################################
/**
* Processes a raw template for conditionals, phrases etc into PHP code for eval()
*
* @param	string	Template
*	@deprecated -- this functionality has been moved to the template API.
* @return	string
*/
function compile_template($template, &$errors = array())
{
	require_once(DIR . '/includes/class_template_parser.php');
	$parser = new vB_TemplateParser($template);

	try
	{
		$parser->validate($errors);
	}
	catch (vB_Exception_TemplateFatalError $e)
	{
		global $vbphrase;
		echo "<p>&nbsp;</p><p>&nbsp;</p>";
		print_form_header('admincp/', '', 0, 1, '', '65%');
		print_table_header($vbphrase['vbulletin_message']);
		print_description_row($vbphrase[$e->getMessage()]);
		print_table_footer(2, construct_button_code($vbphrase['go_back'], 'javascript:history.back(1)'));
		print_cp_footer();
		exit;
	}

	$template = $parser->compile();
	return $template;
}


// #############################################################################
/**
* Returns styles for post editor interface from template
*
* @param	string	Template contents
*
* @return	array
*/
function fetch_posteditor_styles($template)
{
	$item = array();

	preg_match_all('#([a-z0-9-]+):\s*([^\s].*);#siU', $template, $regs);

	foreach ($regs[1] AS $key => $cssname)
	{
		$item[strtolower($cssname)] = trim($regs[2]["$key"]);
	}

	return $item;
}

// #############################################################################
/**
* Prints a row containing an <input type="text" />
*
* @param	string	Title for row
* @param	string	Name for input field
* @param	string	Value for input field
* @param	boolean	Whether or not to htmlspecialchars the input field value
* @param	integer	Size for input field
* @param	integer	Max length for input field
* @param	string	Text direction for input field
* @param	mixed	If specified, overrides the default CSS class for the input field
*/
function print_color_input_row($title, $name, $value = '', $htmlise = true, $size = 35, $maxlength = 0, $direction = '', $inputclass = false)
{
	global $vbulletin, $numcolors;
	$vb5_config = vB::getConfig();

	$direction = verify_text_direction($direction);
	print_label_row(
		$title,
		"<div id=\"ctrl_$name\">
			<input style=\"float:" . vB_Template_Runtime::fetchStyleVar('left') . "; margin-" . vB_Template_Runtime::fetchStyleVar('right') . ": 4px\" type=\"text\" class=\"" . iif($inputclass, $inputclass, 'bginput') . "\" name=\"$name\" id=\"color_$numcolors\" value=\"" . iif($htmlise, htmlspecialchars_uni($value), $value) . "\" size=\"$size\"" . iif($maxlength, " maxlength=\"$maxlength\"") . " dir=\"$direction\" tabindex=\"1\"" . iif($vb5_config['Misc']['debug'], " title=\"name=&quot;$name&quot;\"") . " onchange=\"preview_color($numcolors)\" />
			<div style=\"float:" . vB_Template_Runtime::fetchStyleVar('left') . "\" id=\"preview_$numcolors\" class=\"colorpreview\" onclick=\"open_color_picker($numcolors, event)\"></div>
		</div>",
		'', 'top', $name
	);

	$numcolors++;
}

// #############################################################################
/**
* Builds the color picker popup item for the style editor
*
* @param	integer	Width of each color swatch (pixels)
* @param	string	CSS 'display' parameter (default: 'none')
*
* @return	string
*/
function construct_color_picker($size = 12, $display = 'none')
{
	global $vbulletin, $colorPickerWidth, $colorPickerType;

	$previewsize = 3 * $size;
	$surroundsize = $previewsize * 2;
	$colorPickerWidth = 21 * $size + 22;

	$html = "
	<style type=\"text/css\">
	#colorPicker
	{
		background: black;
		position: absolute;
		left: 0px;
		top: 0px;
		width: {$colorPickerWidth}px;
	}
	#colorFeedback
	{
		border: solid 1px black;
		border-bottom: none;
		width: {$colorPickerWidth}px;
		padding-left: 0;
		padding-right: 0;
	}
	#colorFeedback input
	{
		font: 11px verdana, arial, helvetica, sans-serif;
	}
	#colorFeedback button
	{
		width: 19px;
		height: 19px;
	}
	#txtColor
	{
		border: inset 1px;
		width: 70px;
	}
	#colorSurround
	{
		border: inset 1px;
		white-space: nowrap;
		width: {$surroundsize}px;
		height: 15px;
	}
	#colorSurround td
	{
		background-color: none;
		border: none;
		width: {$previewsize}px;
		height: 15px;
	}
	#swatches
	{
		background-color: black;
		width: {$colorPickerWidth}px;
	}
	#swatches td
	{
		background: black;
		border: none;
		width: {$size}px;
		height: {$size}px;
	}
	</style>
	<div id=\"colorPicker\" style=\"display:$display\" oncontextmenu=\"switch_color_picker(1); return false\" onmousewheel=\"switch_color_picker(event.wheelDelta * -1); return false;\">
	<table id=\"colorFeedback\" class=\"tcat\" cellpadding=\"0\" cellspacing=\"4\" border=\"0\" width=\"100%\">
	<tr>
		<td><button type=\"button\" onclick=\"col_click('transparent'); return false\"><img src=\"" . get_cpstyle_href('colorpicker_transparent.gif') . "\" title=\"'transparent'\" alt=\"\" /></button></td>
		<td>
			<table id=\"colorSurround\" cellpadding=\"0\" cellspacing=\"0\" border=\"0\">
			<tr>
				<td id=\"oldColor\" onclick=\"close_color_picker()\"></td>
				<td id=\"newColor\"></td>
			</tr>
			</table>
		</td>
		<td width=\"100%\"><input id=\"txtColor\" type=\"text\" value=\"\" size=\"8\" /></td>
		<td style=\"white-space:nowrap\">
			<input type=\"hidden\" name=\"colorPickerType\" id=\"colorPickerType\" value=\"$colorPickerType\" />
			<button type=\"button\" onclick=\"switch_color_picker(1); return false\"><img src=\"" . get_cpstyle_href('colorpicker_toggle.gif') . "\" alt=\"\" /></button>
			<button type=\"button\" onclick=\"close_color_picker(); return false\"><img src=\"" . get_cpstyle_href('colorpicker_close.gif') . "\" alt=\"\" /></button>
		</td>
	</tr>
	</table>
	<table id=\"swatches\" cellpadding=\"0\" cellspacing=\"1\" border=\"0\">\n";

	$colors = array(
		'00', '33', '66',
		'99', 'CC', 'FF'
	);

	$specials = array(
		'#000000', '#333333', '#666666',
		'#999999', '#CCCCCC', '#FFFFFF',
		'#FF0000', '#00FF00', '#0000FF',
		'#FFFF00', '#00FFFF', '#FF00FF'
	);

	$green = array(5, 4, 3, 2, 1, 0, 0, 1, 2, 3, 4, 5);
	$blue = array(0, 0, 0, 5, 4, 3, 2, 1, 0, 0, 1, 2, 3, 4, 5, 5, 4, 3, 2, 1, 0);

	for ($y = 0; $y < 12; $y++)
	{
		$html .= "\t<tr>\n";

		$html .= construct_color_picker_element(0, $y, '#000000');
		$html .= construct_color_picker_element(1, $y, $specials["$y"]);
		$html .= construct_color_picker_element(2, $y, '#000000');

		for ($x = 3; $x < 21; $x++)
		{
			$r = floor((20 - $x) / 6) * 2 + floor($y / 6);
			$g = $green["$y"];
			$b = $blue["$x"];

			$html .= construct_color_picker_element($x, $y, '#' . $colors["$r"] . $colors["$g"] . $colors["$b"]);
		}

		$html .= "\t</tr>\n";
	}

	$html .= "\t</table>
	</div>
	<script type=\"text/javascript\">
	<!--
	var tds = fetch_tags(fetch_object(\"swatches\"), \"td\");
	for (var i = 0; i < tds.length; i++)
	{
		tds[i].onclick = swatch_click;
		tds[i].onmouseover = swatch_over;
	}
	//-->
	</script>\n";

	return $html;
}

// #############################################################################
/**
* Builds a single color swatch for the color picker gadget
*
* @param	integer	Current X coordinate
* @param	integer	Current Y coordinate
* @param	string	Color
*
* @return	string
*/
function construct_color_picker_element($x, $y, $color)
{
	return "\t\t<td style=\"background:$color\" id=\"sw$x-$y\"><img src=\"images/clear.gif\" alt=\"\" style=\"width:11px; height:11px\" /></td>\r\n";
}

// #############################################################################
function activateCodeMirror()
{
	//also requires the logic in vbulletin_templatemgr.js which is included as part
	//of the template code that calls this.
	register_js_phrase('fullscreen');
	?>
				<script src="core/clientscript/codemirror/lib/codemirror.js?v=<?php echo SIMPLE_VERSION ?>"></script>

				<script src="core/clientscript/codemirror/mode/xml/xml.js?v=<?php echo SIMPLE_VERSION ?>"></script>
				<script src="core/clientscript/codemirror/mode/javascript/javascript.js?v=<?php echo SIMPLE_VERSION ?>"></script>
				<script src="core/clientscript/codemirror/mode/css/css.js?v=<?php echo SIMPLE_VERSION ?>"></script>
				<!-- <script src="core/clientscript/codemirror/mode/clike/clike.js?v=<?php echo SIMPLE_VERSION ?>"></script> -->
				<script src="core/clientscript/codemirror/mode/htmlmixed/htmlmixed.js?v=<?php echo SIMPLE_VERSION ?>"></script>
				<script src="core/clientscript/codemirror/mode/vbulletin/vbulletin.js?v=<?php echo SIMPLE_VERSION ?>"></script>

				<script src="core/clientscript/codemirror/addon/mode/overlay.js?v=<?php echo SIMPLE_VERSION ?>"></script>
				<script src="core/clientscript/codemirror/addon/selection/active-line.js?v=<?php echo SIMPLE_VERSION ?>"></script>
				<script src="core/clientscript/codemirror/addon/edit/matchbrackets.js?v=<?php echo SIMPLE_VERSION ?>"></script>
				<script src="core/clientscript/codemirror/addon/fold/foldcode.js?v=<?php echo SIMPLE_VERSION ?>"></script>
				<script src="core/clientscript/codemirror/addon/search/search.js?v=<?php echo SIMPLE_VERSION ?>"></script>
				<script src="core/clientscript/codemirror/addon/search/searchcursor.js?v=<?php echo SIMPLE_VERSION ?>"></script>
				<script src="core/clientscript/codemirror/addon/search/match-highlighter.js?v=<?php echo SIMPLE_VERSION ?>"></script>
				<script src="core/clientscript/codemirror/addon/edit/closetag.js?v=<?php echo SIMPLE_VERSION ?>"></script>
				<script src="core/clientscript/codemirror/addon/hint/show-hint.js?v=<?php echo SIMPLE_VERSION ?>"></script>
				<script src="core/clientscript/codemirror/addon/hint/vbulletin-hint.js?v=<?php echo SIMPLE_VERSION ?>"></script>
	<?php
}

function activateCodeMirrorPHP($ids)
{
	$vbphrase = vB_Api::instanceInternal('phrase')->fetch(array('fullscreen'));
	?>
				<script src="core/clientscript/codemirror/lib/codemirror.js?v=<?php echo SIMPLE_VERSION ?>"></script>
				<script src="core/clientscript/codemirror/mode/clike/clike.js?v=<?php echo SIMPLE_VERSION ?>"></script>
				<script src="core/clientscript/codemirror/mode/php/php.js?v=<?php echo SIMPLE_VERSION ?>"></script>

				<script src="core/clientscript/codemirror/addon/selection/active-line.js?v=<?php echo SIMPLE_VERSION ?>"></script>
				<script src="core/clientscript/codemirror/addon/edit/matchbrackets.js?v=<?php echo SIMPLE_VERSION ?>"></script>
				<script src="core/clientscript/codemirror/addon/fold/foldcode.js?v=<?php echo SIMPLE_VERSION ?>"></script>
				<script src="core/clientscript/codemirror/addon/search/match-highlighter.js?v=<?php echo SIMPLE_VERSION ?>"></script>
				<script src="core/clientscript/codemirror/addon/edit/closetag.js?v=<?php echo SIMPLE_VERSION ?>"></script>
				<script src="core/clientscript/codemirror/addon/hint/show-hint.js?v=<?php echo SIMPLE_VERSION ?>"></script>
			<script type="text/javascript">
			<!--
			window.onload = function() {
				$(["<?php echo implode('","', $ids)?>"]).each(function(){
					setUpCodeMirror({
						textarea_id : this,
						phrase_fullscreen : "<?php echo $vbphrase['fullscreen']; ?>",
						mode: "application/x-httpd-php-open"
					})
				});
			};
			//-->
			</script><?php
}

// ###########################################################################################
// START XML STYLE FILE FUNCTIONS

function get_style_export_xml
(
	$styleid,
	$product,
	$product_version,
	$title,
	$mode,
	$remove_guid = false,
	$stylevars_only = false,
	$stylevar_groups = array()
)
{
	global $vbulletin;

	/* Load the master 'style' phrases and then
	build a local $template_groups array using them. */
	$vbphrase = vB_Api::instanceInternal('phrase')->fetchByGroup('style', -1);

	$groups = vB_Library::instance('template')->getTemplateGroupPhrases();
	$template_groups = vB_Api::instanceInternal('phrase')->renderPhrases($groups);
	$template_groups = $template_groups['phrases'];

	$vb5_config = vB::getConfig();
	if (!$vb5_config)
	{
		$vb5_config =& vB::getConfig();
	}

	if ($styleid == -1)
	{
		// set the style title as 'master style'
		$style = array('title' => $vbphrase['master_style']);
		$sqlcondition = "styleid = -1";
		$parentlist = "-1";
		$is_master = true;
	}
	else
	{
		// query everything from the specified style
		$style = $vbulletin->db->query_first("
			SELECT *
			FROM " . TABLE_PREFIX . "style
			WHERE styleid = " . $styleid
		);

		//export as master -- export a style with all changes as a new master style.
		if ($mode == 2)
		{
			//only allowed in debug mode.
			if (!$vb5_config['Misc']['debug'])
			{
				print_cp_no_permission();
			}

			// get all items from this style and all parent styles
			$sqlcondition = "templateid IN(" . implode(',', unserialize($style['templatelist'])) . ")";
			$sqlcondition .= " AND title NOT LIKE 'vbcms_grid_%'";
			$parentlist = $style['parentlist'];
			$is_master = true;
			$title = $vbphrase['master_style'];
		}

		//export with parent styles
		else if ($mode == 1)
		{
			// get all items from this style and all parent styles (except master)
			$sqlcondition = "styleid <> -1 AND templateid IN(" . implode(',', unserialize($style['templatelist'])) . ")";
			//remove the master style id off the end of the list
			$parentlist = substr(trim($style['parentlist']), 0, -3);
			$is_master = false;
		}

		//this style only
		else
		{
			// get only items customized in THIS style
			$sqlcondition = "styleid = " . $styleid;
			$parentlist = $styleid;
			$is_master = false;
		}
	}

	if ($product == 'vbulletin')
	{
		$sqlcondition .= " AND (product = '" . vB::getDbAssertor()->escape_string($product) . "' OR product = '')";
	}
	else
	{
		$sqlcondition .= " AND product = '" . vB::getDbAssertor()->escape_string($product) . "'";
	}

	// set a default title
	if ($title == '' OR $styleid == -1)
	{
		$title = $style['title'];
	}

	if (!empty($style['dateline']))
	{
		$dateline = $style['dateline'];
	}
	else
	{
		$dateline = vB::getRequest()->getTimeNow();
	}

	// --------------------------------------------
	// query the templates and put them in an array

	$templates = array();

	if (!$stylevars_only)
	{
		$gettemplates = $vbulletin->db->query_read("
			SELECT title, templatetype, username, dateline, version,
			IF(templatetype = 'template', template_un, template) AS template,
			textonly
			FROM " . TABLE_PREFIX . "template
			WHERE $sqlcondition
			ORDER BY title
		");

		$ugcount = $ugtemplates = 0;
		while ($gettemplate = $vbulletin->db->fetch_array($gettemplates))
		{
			switch($gettemplate['templatetype'])
			{
				case 'template': // regular template

					// if we have ad template, and we are exporting as master, make sure we do not export the ad data
					if (substr($gettemplate['title'], 0, 3) == 'ad_' AND $mode == 2)
					{
						$gettemplate['template'] = '';
					}

					$isgrouped = false;
					foreach(array_keys($template_groups) AS $group)
					{
						if (strpos(strtolower(" $gettemplate[title]"), $group) == 1)
						{
							$templates["$group"][] = $gettemplate;
							$isgrouped = true;
						}
					}
					if (!$isgrouped)
					{
						if ($ugtemplates % 10 == 0)
						{
							$ugcount++;
						}
						$ugtemplates++;
						//sort ungrouped templates last.
						$ugcount_key = 'zzz' . str_pad($ugcount, 5, '0', STR_PAD_LEFT);
						$templates[$ugcount_key][] = $gettemplate;
						$template_groups[$ugcount_key] = construct_phrase($vbphrase['ungrouped_templates_x'], $ugcount);
					}
				break;

				case 'stylevar': // stylevar
					$templates[$vbphrase['stylevar_special_templates']][] = $gettemplate;
				break;

				case 'css': // css
					$templates[$vbphrase['css_special_templates']][] = $gettemplate;
				break;

				case 'replacement': // replacement
					$templates[$vbphrase['replacement_var_special_templates']][] = $gettemplate;
				break;
			}
		}
		unset($gettemplate);
		$vbulletin->db->free_result($gettemplates);
		if (!empty($templates))
		{
			ksort($templates);
		}

	}

	// --------------------------------------------
	// fetch stylevar-dfns

	$stylevarinfo = get_stylevars_for_export($product, $parentlist, $stylevar_groups);
	$stylevar_cache = $stylevarinfo['stylevars'];
	$stylevar_dfn_cache = $stylevarinfo['stylevardfns'];

	if (empty($templates) AND empty($stylevar_cache) AND empty($stylevar_dfn_cache))
	{
		throw new vB_Exception_AdminStopMessage('download_contains_no_customizations');
	}

	// --------------------------------------------
	// now output the XML

	$xml = new vB_XML_Builder();
	$rootAttributes =
		array(
			'name' => $title,
			'vbversion' => $product_version,
			'product' => $product,
			'type' => $is_master ? 'master' : 'custom',
			'dateline' => $dateline,
		);
	if (isset($style['styleattributes']) AND $style['styleattributes'] != vB_Library_Style::ATTR_DEFAULT)
	{
		$rootAttributes['styleattributes'] = $style['styleattributes'];
	}
	$xml->add_group('style',
		$rootAttributes
	);


	/*
	 * Check if it's a THEME, and add extra guid, icon & previewimage tags.
	 */
	if (!empty($style['guid']))
	{
		// we allow removing the GUID
		if (!$remove_guid)
		{
			$xml->add_tag('guid', $style['guid']);
		}

		// optional, image data
		$optionalImages = array(
			// DB column name => XML tag name
			'filedataid' => 'icon',
			'previewfiledataid' => 'previewimage',
		);
		foreach ($optionalImages AS $dbColumn => $tagname)
		{
			if (!empty($style[$dbColumn]))
			{
				$filedata = vB_Api::instanceInternal('filedata')->fetchImageByFiledataid($style[$dbColumn]);
				if (!empty($filedata['filedata']))
				{
					$xml->add_tag($tagname, base64_encode($filedata['filedata']));
				}
			}
		}
	}

	foreach($templates AS $group => $grouptemplates)
	{
		$xml->add_group('templategroup', array('name' => iif(isset($template_groups["$group"]), $template_groups["$group"], $group)));
		foreach($grouptemplates AS $template)
		{
			$attributes = array(
				'name' => htmlspecialchars($template['title']),
				'templatetype' => $template['templatetype'],
				'date' => $template['dateline'],
				'username' => $template['username'],
				'version' => htmlspecialchars_uni($template['version']),
			);
			$textonly = !empty($template['textonly']);
			if ($textonly)
			{
				$attributes['textonly'] = 1;
			}

			$xml->add_tag('template', $template['template'], $attributes, true);
		}
		$xml->close_group();
	}

	$xml->add_group('stylevardfns');
	foreach ($stylevar_dfn_cache AS $stylevargroupname => $stylevargroup)
	{
		$xml->add_group('stylevargroup', array('name' => $stylevargroupname));
		foreach($stylevargroup AS $stylevar)
		{
			$xml->add_tag('stylevar', '',
				array(
					'name' => htmlspecialchars($stylevar['stylevarid']),
					'datatype' => $stylevar['datatype'],
					'validation' => base64_encode($stylevar['validation']),
					'failsafe' => base64_encode($stylevar['failsafe'])
				)
			);
		}
		$xml->close_group();
	}
	$xml->close_group();

	$xml->add_group('stylevars');
	foreach ($stylevar_cache AS $stylevarid => $stylevar)
	{
		$xml->add_tag('stylevar', '',
			array(
				'name' => htmlspecialchars($stylevar['stylevarid']),
				'value' => base64_encode($stylevar['value'])
			)
		);
	}
	$xml->close_group();

	$xml->close_group();

	$doc = "<?xml version=\"1.0\" encoding=\"ISO-8859-1\"?>\r\n\r\n";
	$doc .= $xml->output();
	$xml = null;
	return $doc;
}

/// #############################################################################
/**
* Reads XML style file and imports data from it into the database
*
* @param	string	$xml		XML data
* @param	integer	$styleid	Style ID
* @param	integer	$parentid	Parent style ID
* @param	string	$title		New style title
* @param	boolean	$anyversion	Allow vBulletin version mismatch
* @param	integer	$displayorder	Display order for new style
* @param	boolean	$userselct	Allow user selection of new style
* @param  	int|null	$startat	Starting template group index for this run of importing templates (0 based). Null means all templates (single run)
* @param  	int|null	$perpage	Number of templates to import at a time
* @param	boolean	$silent		Run silently (do not echo)
* @param	array|boolean	$parsed_xml	Parsed array of XML data. If provided the function will ignore $xml and use the provided, already parsed data.
*
* @return	array	Array of information about the imported style
*/
function xml_import_style(
	$xml,
	$styleid = -1,
	$parentid = -1,
	$title = '',
	$anyversion = false,
	$displayorder = 1,
	$userselect = true,
	$startat = null,
	$perpage = null,
	$silent = false,
	$parsed_xml = false,
	$requireUniqueTitle = true
)
{
	//checking the root node name
	if (!empty($xml))
	{
		$r = new XMLReader();
		if ($r->xml($xml))
		{
			if ($r->read())
			{
				$node_name = $r->name;
				if ($node_name != 'style')
				{
					print_stop_message2('file_uploaded_not_in_right_format_error');
				}
			}
			else
			{
				//can not read the document
				print_stop_message2('file_uploaded_unreadable');
			}
		}
		else
		{
			//can not open the xml
			print_stop_message2('file_uploaded_unreadable');
		}
	}

	if (!$silent)
	{
		$vbphrase = vB_Api::instanceInternal('phrase')->fetch(array('importing_style', 'please_wait', 'creating_a_new_style_called_x'));
		print_dots_start('<b>' . $vbphrase['importing_style'] . "</b>, $vbphrase[please_wait]", ':', 'dspan');
	}

	if (empty($parsed_xml))
	{
		//where is this used?  I hate having this random global value in the middle of this function
		$xmlobj = new vB_XML_Parser($xml);
		if ($xmlobj->error_no())
		{
			if ($silent)
			{
				throw new vB_Exception_AdminStopMessage('no_xml_and_no_path');
			}
			print_dots_stop();
			print_stop_message2('no_xml_and_no_path');
		}

		if(!$parsed_xml = $xmlobj->parse())
		{
			if ($silent)
			{
				throw new vB_Exception_AdminStopMessage(array('xml_error_x_at_line_y', $xmlobj->error_string(), $xmlobj->error_line()));
			}
			print_dots_stop();
			print_stop_message2(array('xml_error_x_at_line_y', $xmlobj->error_string(), $xmlobj->error_line()));
		}
	}

	$version = $parsed_xml['vbversion'];
	$master = ($parsed_xml['type'] == 'master' ? 1 : 0);
	$title = (empty($title) ? $parsed_xml['name'] : $title);
	$product = (empty($parsed_xml['product']) ? 'vbulletin' : $parsed_xml['product']);
	$styleattributes = (isset($parsed_xml['styleattributes']) ? intval($parsed_xml['styleattributes']) : vB_Library_Style::ATTR_DEFAULT);
	$dateline = (isset($parsed_xml['dateline']) ? intval($parsed_xml['dateline']) : vB::getRequest()->getTimeNow());
	$assertor = vB::getDbAssertor(); // cache assertor to avoid repeated, unnecessary function call.


	$one_pass = (is_null($startat) AND is_null($perpage));
	if (!$one_pass AND (!is_numeric($startat) OR !is_numeric($perpage) OR $perpage <= 0 OR $startat < 0))
	{
			if ($silent)
			{
				throw new vB_Exception_AdminStopMessage('');
			}
			print_dots_stop();
			print_stop_message2('');
	}

	$outputtext = '';
	if ($one_pass OR ($startat == 0))
	{
		require_once(DIR . '/includes/adminfunctions.php');
		// version check
		$full_product_info = fetch_product_list(true);
		$product_info = $full_product_info["$product"];

		if ($version != $product_info['version'] AND !$anyversion AND !$master)
		{
			if ($silent)
			{
				throw new vB_Exception_AdminStopMessage(array('upload_file_created_with_different_version', $product_info['version'], $version));
			}
			print_dots_stop();
			print_stop_message2(array('upload_file_created_with_different_version', $product_info['version'], $version));
		}

		//Initialize the style -- either init the master, create a new style, or verify the style to overwrite.
		if ($master)
		{
			$import_data = @unserialize(fetch_adminutil_text('master_style_import'));
			if (!empty($import_data) AND (TIMENOW - $import_data['last_import']) <= 30)
			{
				if ($silent)
				{
					throw new vB_Exception_AdminStopMessage(array('must_wait_x_seconds_master_style_import', vb_number_format($import_data['last_import'] + 30 - TIMENOW)));
				}
				print_dots_stop();
				print_stop_message2(array('must_wait_x_seconds_master_style_import',  vb_number_format($import_data['last_import'] + 30 - TIMENOW)));
			}

			$products = array($product);
			if ($product == 'vbulletin')
			{
				$products[] = '';
			}
			$assertor->assertQuery('vBForum:deleteProductTemplates', array('products' =>$products));
			$assertor->assertQuery('vBForum:updateProductTemplates', array('products' =>$products));
			$styleid = -1;
		}
		else
		{
			if ($styleid == -1)
			{
				// creating a new style
				if ($requireUniqueTitle AND $assertor->getRow('style', array('title' => $title)))
				{
					if ($silent)
					{
						throw new vB_Exception_AdminStopMessage(array('style_already_exists', $title));
					}
					print_dots_stop();
					print_stop_message2(array('style_already_exists',  $title));
				}
				else
				{
					if (!$silent)
					{
						$outputtext = construct_phrase($vbphrase['creating_a_new_style_called_x'], $title) . "<br>\n";
					}

					/*insert query*/
					$styleid = $assertor->insert('style', [
						'title' => $title,
						'parentid' => $parentid,
						'displayorder' => $displayorder,
						'userselect' => $userselect ? 1 : 0,
						'styleattributes' => $styleattributes,
						'dateline' => $dateline,
					]);

					if (is_array($styleid))
					{
						$styleid = array_pop($styleid);
					}
				}
			}
			else
			{
				// overwriting an existing style
				if ($oldStyleData = $assertor->getRow('style', array('styleid' => $styleid)))
				{
					/*
						Do an update if needed.
						Especially required for forcing theme XML changes to stick during upgrade
						(ex adding/changing styleattributes)
					*/
					$changed = (
						$oldStyleData['title'] != $title ||
						$oldStyleData['parentid'] != $parentid ||
						$oldStyleData['displayorder'] != $displayorder ||
						$oldStyleData['userselect'] != $userselect ||
						$oldStyleData['styleattributes'] != $styleattributes ||
						$oldStyleData['dateline'] != $dateline
					);

					if ($changed)
					{
						$styleid = $assertor->update('style',
							[
								'title' => $title,
								'parentid' => $parentid,
								'displayorder' => $displayorder,
								'userselect' => $userselect ? 1 : 0,
								'styleattributes' => $styleattributes,
								'dateline' => $dateline,
							],
							['styleid' => $styleid]
						);
					}
				}
				else
				{
					if ($silent)
					{
						throw new vB_Exception_AdminStopMessage('cant_overwrite_non_existent_style');
					}
					print_dots_stop();
					print_stop_message2('cant_overwrite_non_existent_style');
				}
			}
		}
	}
	else
	{
		//We should never get styleid = -1 unless $master is true;
		if (($styleid == -1) AND !$master)
		{
			// According to this code, a style's title is a unique identifier (why not use guid?). This might be problematic.
			$stylerec = $assertor->getRow('style', array('title' => $title));

			if ($stylerec AND intval($stylerec['styleid']))
			{
				$styleid = $stylerec['styleid'];
			}
			else
			{
				if ($silent)
				{
					throw new vB_Exception_AdminStopMessage(array('incorrect_style_setting', $title));
				}
				print_dots_stop();
				print_stop_message2(array('incorrect_style_setting',  $title));
			}
		}
	}

	//load the templates
	$arr = vB_XML_Parser::getList($parsed_xml, 'templategroup');
	if ($arr)
	{
		$templates_done = (is_numeric($startat) AND (count($arr) <= $startat));
		if ($one_pass OR !$templates_done)
		{
			if (!$one_pass)
			{
				$arr = array_slice($arr, $startat, $perpage);
			}
			$outputtext .= xml_import_template_groups($styleid, $product, $arr, !$one_pass);
		}
	}
	else
	{
		$templates_done = true;
	}

	//note that templates may actually be done at this point, but templates_done is
	//only true if templates were completed in a prior step. If we are doing a multi-pass
	//process, we don't want to install stylevars in the same pass.  We aren't really done
	//until we hit a pass where the templates are done before processing.
	$done = ($one_pass OR $templates_done);
	if ($done)
	{
		//load stylevars and definitions
		// re-import any stylevar definitions
		if ($master AND !empty($parsed_xml['stylevardfns']['stylevargroup']))
		{
			xml_import_stylevar_definitions($parsed_xml['stylevardfns'], 'vbulletin');
		}

		//if the tag is present but empty we'll end up with a string with whitespace which
		//is a non "empty" value.
		if (!empty($parsed_xml['stylevars']) AND is_array($parsed_xml['stylevars']))
		{
			xml_import_stylevars($parsed_xml['stylevars'], $styleid);
		}

		if ($master)
		{
			xml_import_restore_ad_templates();
			build_adminutil_text('master_style_import', serialize(array('last_import' => TIMENOW)));
		}
		if (!$silent)
		{
			print_dots_stop();
		}
	}
	$fastDs = vB_FastDS::instance();

	//We want to force a fastDS rebuild, but we can't just call rebuild. There may be dual web servers,
	// and calling rebuild only rebuilds one of them.
	$options = vB::getDatastore()->getValue('miscoptions');
	$options['tmtdate'] = vB::getRequest()->getTimeNow();
	vB::getDatastore()->build('miscoptions', serialize($options), 1);

	return array(
		'version' => $version,
		'master'  => $master,
		'title'   => $title,
		'product' => $product,
		'done'    => $done,
		'overwritestyleid' => $styleid,
		'output'  => $outputtext,
	);
}

function xml_import_template_groups($styleid, $product, $templategroup_array, $output_group_name, $printInfo = true)
{
	global $vbphrase;

	$isinstaller = (defined('VB_AREA') AND (VB_AREA == 'Upgrade' OR VB_AREA == 'Install'));
	$doOutput = ($printInfo AND !$isinstaller);

	$safe_product =  vB::getDbAssertor()->escape_string($product);

	$querytemplates = 0;
	$outputtext = '';
	if ($doOutput)
	{
		echo defined('NO_IMPORT_DOTS') ? "\n" : '<br />';
		vbflush();
	}
	foreach ($templategroup_array AS $templategroup)
	{
		if (empty($templategroup['template'][0]))
		{
			$tg = array($templategroup['template']);
		}
		else
		{
			$tg = &$templategroup['template'];
		}

		if ($output_group_name)
		{
			$text = construct_phrase($vbphrase['template_group_x'], $templategroup['name']);
			$outputtext .= $text;
			if ($doOutput)
			{
				echo $text;
				vbflush();
			}
		}

		foreach($tg AS $template)
		{
			$textonly = !empty($template['textonly']);
			// Skip compile for textonly templates or non-template templatetypes.
			if ($textonly OR $template['templatetype'] != 'template')
			{
				$parsedTemplate = $template['value'];
			}
			else
			{
				$parsedTemplate = compile_template($template['value']);
			}

			$querybit = array(
				'styleid'     => $styleid,
				'title'       => $template['name'],
				'template'    => $template['templatetype'] == 'template' ? $parsedTemplate : $template['value'],
				'template_un' => $template['templatetype'] == 'template' ? $template['value'] : '',
				'dateline'    => $template['date'],
				'username'    => $template['username'],
				'version'     => $template['version'],
				'product'     => $product,
				'textonly'    => $textonly,
			);
			$querybit['templatetype'] = $template['templatetype'];

			$querybits[] = $querybit;

			if (++$querytemplates % 10 == 0 OR $templategroup['name'] == 'Css')
			{
				/*insert query*/
				vB::getDbAssertor()->assertQuery('replaceTemplates', array('querybits' => $querybits));
				$querybits = array();
			}

			// Send some output to the browser inside this loop so certain hosts
			// don't artificially kill the script. See bug #34585
			if (!defined('SUPPRESS_KEEPALIVE_ECHO'))
			{
				echo ($isinstaller ? ' ' : '-');
				vbflush();
			}
		}

		if ($doOutput)
		{
			echo defined('NO_IMPORT_DOTS') ? "\n" : '<br />';
			vbflush();
		}
	}

	// insert any remaining templates
	if (!empty($querybits))
	{
		vB::getDbAssertor()->assertQuery('replaceTemplates', array('querybits' => $querybits));
		$querybits = array();
	}

	return $outputtext;
}

function xml_import_restore_ad_templates()
{
	// Get the template titles
	$save = array();
	$save_tables = vB::getDbAssertor()->assertQuery('template', array(vB_dB_Query::CONDITIONS_KEY=> array(
			array('field'=>'templatetype', 'value' => 'template', vB_dB_Query::OPERATOR_KEY => vB_dB_Query::OPERATOR_EQ),
			array('field'=>'styleid', 'value' => -10, vB_dB_Query::OPERATOR_KEY => vB_dB_Query::OPERATOR_EQ),
			array('field'=>'product', 'value' => array('vbulletin', ''), vB_dB_Query::OPERATOR_KEY => vB_dB_Query::OPERATOR_EQ),
			array('field'=>'title', 'value' => 'ad_', vB_dB_Query::OPERATOR_KEY => vB_dB_Query::OPERATOR_BEGINS),
	)));


	foreach ($save_tables as $table)
	{
		$save[] = $table['title'];
	}

	// Are there any
	if (count($save))
	{
		// Delete any style id -1 ad templates that may of just been imported.
		vB::getDbAssertor()->delete('template', array(
			array('field'=>'templatetype', 'value' => 'template', vB_dB_Query::OPERATOR_KEY => vB_dB_Query::OPERATOR_EQ),
			array('field'=>'styleid', 'value' => -1, vB_dB_Query::OPERATOR_KEY => vB_dB_Query::OPERATOR_EQ),
			array('field'=>'product', 'value' => array('vbulletin', ''), vB_dB_Query::OPERATOR_KEY => vB_dB_Query::OPERATOR_EQ),
			array('field'=>'title', 'value' => $save, vB_dB_Query::OPERATOR_KEY => vB_dB_Query::OPERATOR_EQ),
		));


		// Replace the -1 templates with the -10 before they are deleted
		vB::getDbAssertor()->assertQuery('template',
			array(
				vB_dB_Query::TYPE_KEY => vB_dB_Query::QUERY_UPDATE,
				vB_dB_Query::CONDITIONS_KEY => array(
					array('field'=>'templatetype', 'value' => 'template', vB_dB_Query::OPERATOR_KEY => vB_dB_Query::OPERATOR_EQ),
					array('field'=>'styleid', 'value' => -10, vB_dB_Query::OPERATOR_KEY => vB_dB_Query::OPERATOR_EQ),
					array('field'=>'product', 'value' => array('vbulletin', ''), vB_dB_Query::OPERATOR_KEY => vB_dB_Query::OPERATOR_EQ),
					array('field'=>'title', 'value' => $save, vB_dB_Query::OPERATOR_KEY => vB_dB_Query::OPERATOR_EQ),
				),
				'styleid' => -1,
			)
		);

	}
}

function xml_import_stylevar_definitions($stylevardfns, $product)
{
	global $vbulletin;

	$querybits = [];
	$stylevardfns = vB_XML_Parser::getList($stylevardfns, 'stylevargroup');

	/*
		Delete the existing stylevars
		parentid will = 0 for imported stylevars,
		but is set to -1 for custom added sytlevars.
		We only really care about this for default
		vbulletin as any other products will clear up
		their own stylevars when they are uninstalled.
	*/

	if ($product == 'vbulletin')
	{
		$where = array('product' => 'vbulletin', 'parentid' => 0);
	}
	else
	{
		$where = array('product' => $product);
	}

	vB::getDbAssertor()->delete('vBForum:stylevardfn', $where);

	foreach ($stylevardfns AS $stylevardfn_group)
	{
		$sg = vB_XML_Parser::getList($stylevardfn_group, 'stylevar');
		foreach ($sg AS $stylevardfn)
		{
			$querybits[] = "('" . $vbulletin->db->escape_string($stylevardfn['name']) . "', -1, '" .
				$vbulletin->db->escape_string($stylevardfn_group['name']) . "', '" .
				$vbulletin->db->escape_string($product) . "', '" .
				$vbulletin->db->escape_string($stylevardfn['datatype']) . "', '" .
				$vbulletin->db->escape_string(base64_decode($stylevardfn['validation'])) . "', '" .
				$vbulletin->db->escape_string(base64_decode($stylevardfn['failsafe'])) . "', 0, 0
			)";
		}

		if (!empty($querybits))
		{
			$vbulletin->db->query_write("
				REPLACE INTO " . TABLE_PREFIX . "stylevardfn
				(stylevarid, styleid, stylevargroup, product, datatype, validation, failsafe, parentid, parentlist)
				VALUES
				" . implode(',', $querybits) . "
			");
		}
		$querybits = array();
	}
}

function xml_import_stylevars($stylevars, $styleid)
{
	$values = array();
	$sv = vB_XML_Parser::getList($stylevars, 'stylevar');

	foreach ($sv AS $stylevar)
	{
		//the parser merges attributes and child nodes into a single array.  The unnamed text
		//children get placed into a key called "value" automagically.  Since we don't have any
		//text children we just take the first one.
		$values[] = array(
			'stylevarid' => $stylevar['name'],
			'styleid' => $styleid,
			'value' => base64_decode($stylevar['value'][0]),
			'dateline' => time(),
			'username' => 'Style-Importer',
		);
	}

	if (!empty($values))
	{
		vB::getDbAssertor()->assertQuery('replaceValues', array('table' => 'stylevar', 'values' => $values));
	}
}


/**
*	Get the stylevar list processed to export
*
*	Seperated into its own function for reuse by products
*
*	@param string product -- The name of the product to
*	@param string stylelist -- The styles to export as a comma seperated string
*		(in descending order of precedence).  THE CALLER IS RESPONSIBLE FOR SANITIZING THE
*		INPUT.
*/
function get_stylevars_for_export($product, $stylelist, $stylevar_groups = array())
{
	$assertor = vB::getDbAssertor();
	$queryParams = array(
		'product'   => ($product == 'vbulletin') ? array('vbulletin', '') : array((string)$product),
		'stylelist' => explode(',', $stylelist),
		'stylevar_groups' => $stylevar_groups,
	);

	$stylevar_cache = array();
	$stylevars = $assertor->getRows('vBForum:getStylevarsForExport', $queryParams);
	foreach ($stylevars AS $stylevar)
	{
		$stylevar_cache[$stylevar['stylevarid']] = $stylevar;
		ksort($stylevar_cache);
	}

	$stylevar_dfn_cache = array();
	$stylevar_dfns = $assertor->getRows('vBForum:getStylevarsDfnForExport', $queryParams);
	foreach ($stylevar_dfns AS $stylevar_dfn)
	{
		$stylevar_dfn_cache[$stylevar_dfn['stylevargroup']][] = $stylevar_dfn;
	}

	return array("stylevars" => $stylevar_cache, "stylevardfns" => $stylevar_dfn_cache);
}


// #############################################################################
/**
* Converts a version number string into an array that can be parsed
* to determine if which of several version strings is the newest.
*
* @param	string	Version string to parse
*
* @return	array	Array of 6 bits, in decreasing order of influence; a higher bit value is newer
*/
function fetch_version_array($version)
{
	// parse for a main and subversion
	if (preg_match('#^([a-z]+ )?([0-9\.]+)[\s-]*([a-z].*)$#i', trim($version), $match))
	{
		$main_version = $match[2];
		$sub_version = $match[3];
	}
	else
	{
		$main_version = $version;
		$sub_version = '';
	}

	$version_bits = explode('.', $main_version);

	// pad the main version to 4 parts (1.1.1.1)
	if (sizeof($version_bits) < 4)
	{
		for ($i = sizeof($version_bits); $i < 4; $i++)
		{
			$version_bits["$i"] = 0;
		}
	}

	// default sub-versions
	$version_bits[4] = 0; // for alpha, beta, rc, pl, etc
	$version_bits[5] = 0; // alpha, beta, etc number

	if (!empty($sub_version))
	{
		// match the sub-version
		if (preg_match('#^(A|ALPHA|B|BETA|G|GAMMA|RC|RELEASE CANDIDATE|GOLD|STABLE|FINAL|PL|PATCH LEVEL)\s*(\d*)\D*$#i', $sub_version, $match))
		{
			switch (strtoupper($match[1]))
			{
				case 'A':
				case 'ALPHA';
					$version_bits[4] = -4;
					break;

				case 'B':
				case 'BETA':
					$version_bits[4] = -3;
					break;

				case 'G':
				case 'GAMMA':
					$version_bits[4] = -2;
					break;

				case 'RC':
				case 'RELEASE CANDIDATE':
					$version_bits[4] = -1;
					break;

				case 'PL':
				case 'PATCH LEVEL';
					$version_bits[4] = 1;
					break;

				case 'GOLD':
				case 'STABLE':
				case 'FINAL':
				default:
					$version_bits[4] = 0;
					break;
			}

			$version_bits[5] = $match[2];
		}
	}

	// sanity check -- make sure each bit is an int
	for ($i = 0; $i <= 5; $i++)
	{
		$version_bits["$i"] = intval($version_bits["$i"]);
	}

	return $version_bits;
}

/**
* Compares two version strings. Returns true if the first parameter is
* newer than the second.
*
* @param	string	Version string; usually the latest version
* @param	string	Version string; usually the current version
* @param	bool	Flag to allow check if the versions are the same
*
* @return	bool	True if the first argument is newer than the second, or if 'check_same' is true and the versions are the equal
*/
function is_newer_version($new_version_str, $cur_version_str, $check_same = false)
{
	// if they're the same, don't even bother
	if ($cur_version_str != $new_version_str)
	{
		$cur_version = fetch_version_array($cur_version_str);
		$new_version = fetch_version_array($new_version_str);

		// iterate parts
		for ($i = 0; $i <= 5; $i++)
		{
			if ($new_version["$i"] != $cur_version["$i"])
			{
				// true if newer is greater
				return ($new_version["$i"] > $cur_version["$i"]);
			}
		}
	}
	else if ($check_same)
	{
		return true;
	}

	return false;
}

/**
* Function used for usort'ing a collection of templates.
* This function will return newer versions first.
*
* @param	array	First version
* @param	array	Second version
*
* @return	integer	-1, 0, 1
*/
function history_compare($a, $b)
{
	// if either of them does not have a version, make it look really old to the
	// comparison tool so it doesn't get bumped all the way up when its not supposed to
	if (!$a['version'])
	{
		$a['version'] = "0.0.0";
	}

	if (!$b['version'])
	{
		$b['version'] = "0.0.0";
	}

	// these return values are backwards to sort in descending order
	if (is_newer_version($a['version'], $b['version']))
	{
		return -1;
	}
	else if (is_newer_version($b['version'], $a['version']))
	{
		return 1;
	}
	else
	{
		if($a['type'] == $b['type'])
		{
			return ($a['dateline'] > $b['dateline']) ? -1 : 1;
		}
		else if($a['type'] == "historical")
		{
			return 1;
		}
		else
		{
			return -1;
		}
	}
}

/**
* Fetches a current or historical template.
*
* @param	integer	The ID (in the appropriate table) of the record you want to fetch
* @param	string	Type of template you want to fetch; should be "current" or "historical"
*
* @return	array	The data for the matching record
*/
function fetch_template_current_historical(&$id, $type)
{
	global $vbulletin;

	$id = intval($id);

	if ($type == 'current')
	{
		return $vbulletin->db->query_first("
			SELECT *, template_un AS templatetext
			FROM " . TABLE_PREFIX . "template
			WHERE templateid = $id
		");
	}
	else
	{
		return $vbulletin->db->query_first("
			SELECT *, template AS templatetext
			FROM " . TABLE_PREFIX . "templatehistory
			WHERE templatehistoryid = $id
		");
	}
}


/**
* Fetches the list of templates that have a changed status in the database
*
* List is hierarchical by style.
*
* @return array Associative array of styleid => template list with each template
* list being an array of templateid => template record.
*/
function fetch_changed_templates()
{
	$templates = [];
	$set = vB::getDbAssertor()->assertQuery('vBForum:fetchchangedtemplates', []);
	foreach ($set AS $template)
	{
		$templates[$template['styleid']][$template['templateid']] = $template;
	}
	return $templates;
}

/**
* Fetches the count templates that have a changed status in the database
*
* @return int Number of changed templates
*/
function fetch_changed_templates_count()
{
	$result = vB::getDbAssertor()->getRow('vBForum:getChangedTemplatesCount');
	return $result['count'];
}

/**
*	Get the template from the template id
*
*	@param id template id
* @return array template table record
*/
function fetch_template_by_id($id)
{
	$filter = array('templateid' => intval($id));
	return fetch_template_internal($filter);
}

/**
*	Get the template from the template using the style and title
*
*	@param 	int 	styleid
* 	@param  string	title
* 	@return array 	template table record
*/
function fetch_template_by_title($styleid, $title)
{
	$filter = array('styleid' => intval($styleid), 'title' => (string) $title, 'templatetype' => 'template');
	return fetch_template_internal($filter);
}


/**
*	Get the template from the templatemerge (saved origin templates in the merge process)
* using the id
*
* The record is returned with the addition of an extra template_un field.
* This is set to the same value as the template field and is intended to match up the
* fields in the merge table with the fields in the main template table.
*
*	@param 	int 	id - Note that this is the same value as the main template table id
* 	@return array 	template record with extra template_un field
*/
function fetch_origin_template_by_id($id)
{
	$result = vB::getDbAssertor()->getRow('templatemerge', array('templateid' => intval($id)));

	if ($result)
	{
		$result['template_un'] = $result['template'];
	}
	return $result;
}

/**
*	Get the template from the template using the id
*
* The record is returned with the addition of an extra template_un field.
* This is set to the same value as the template field and is intended to match up the
* fields in the merge table with the fields in the main template table.
*
*	@param int id - Note that this is the not same value as the main template table id,
*		there can be multiple saved history versions for a given template
* @return array template record with extra template_un field
*/
function fetch_historical_template_by_id($id)
{
	$result = vB::getDbAssertor()->getRow('templatehistory', array('templatehistoryid' => intval($id)));

	//adjust to look like the main template result
	if ($result)
	{
		$result['template_un'] = $result['template'];
	}
	return $result;
}

/**
*	Get the template record
*
* This should only be called by cover functions in the file
* caller is responsible for sql security on $filter;
*
*	@filter Array	Filters to be used in the where clause. Field should be the key:
*					e.g: array('templateid' => $someValue)
* @private
*/
function fetch_template_internal($filter)
{
	$assertor = vB::getDbAssertor();
	$structure = $assertor->fetchTableStructure('template');
	$structure = $structure['structure'];

	$queryParams = array();
	foreach ($filter AS $field => $val)
	{
		if (in_array($field, $structure))
		{
			$queryParams[$field] = $val;
		}
	}

	return $assertor->getRow('template', $queryParams);
}


/**
* Get the requested templates for a merge operation
*
*	This gets the templates needed to show the merge display for a given custom
* template.  These are the custom template, the current default template, and the
* origin template saved when the template was initially merged.
*
* We can only display merges for templates that were actually merged during upgrade
*	as we only save the necesary information at that point.  If we don't have the
* available inforamtion to support the merge display, then an exception will be thrown
* with an explanatory message. Updating a template after upgrade
*
*	If the custom template was successfully merged we return the historical template
* save at upgrade time instead of the current (automatically updated at merge time)
* template.  Otherwise the differences merged into the current template will not be
* correctly displayed.
*
*	@param int templateid - The id of the custom user template to start this off
*	@throws Exception thrown if state does not support a merge display for
* 	the requested template
*	@return array array('custom' => $custom, 'new' => $new, 'origin' => $origin)
*/
function fetch_templates_for_merge($templateid)
{
	global $vbphrase;
	if (!$templateid)
	{
		throw new Exception($vbphrase['merge_error_invalid_template']);
	}

	$custom = fetch_template_by_id($templateid);
	if (!$custom)
	{
		throw new Exception(construct_phrase($vbphrase['merge_error_notemplate'], $templateid));
	}

	if ($custom['mergestatus'] == 'none')
	{
		throw new Exception($vbphrase['merge_error_nomerge']);
	}

	$new = fetch_template_by_title(-1, $custom['title']);
	if (!$new)
	{
		throw new Exception(construct_phrase($vbphrase['merge_error_nodefault'],  $custom['title']));
	}

	$origin = fetch_origin_template_by_id($custom['templateid']);
	if (!$origin)
	{
		throw new Exception(construct_phrase($vbphrase['merge_error_noorigin'],  $custom['title']));
	}

	if ($custom['mergestatus'] == 'merged')
	{
		$custom = fetch_historical_template_by_id($origin['savedtemplateid']);
		if (!$custom)
		{
			throw new Exception(construct_phrase($vbphrase['merge_error_nohistory'],  $custom['title']));
		}
	}

	return array('custom' => $custom, 'new' => $new, 'origin' => $origin);
}


/**
* Format the text for a merge conflict
*
* Take the three conflict text strings and format them into a human readable
* text block for display.
*
* @param string	Text from custom template
* @param string	Text from origin template
* @param string	Text from current VBulletin template
* @param string	Version string for origin template
* @param string	Version string for currnet VBulletin template
* @param bool	Whether to output the wrapping text with html markup for richer display
*
* @return string -- combined text
*/
function format_conflict_text($custom, $origin, $new, $origin_version, $new_version, $html_markup = false, $wrap = true)
{
	$phrases = vB_Api::instanceInternal('phrase')->renderPhrases([
		'new_default_value',
		'old_default_value',
		'your_customized_value',
	]);
	list($new_title, $origin_title, $custom_title) = $phrases['phrases'];

	if ($html_markup)
	{
		$text =
			"<div class=\"merge-conflict-row\"><b>$custom_title</b><div>" . format_diff_text($custom, $wrap) . "</div></div>"
			. "<div class=\"merge-conflict-row\"><b>$origin_title</b><div>" . format_diff_text($origin, $wrap) . "</div></div>"
			. "<div class=\"merge-conflict-final-row\"><b>$new_title</b><div>" . format_diff_text($new, $wrap) . "</div></div>";
	}
	else
	{
		$origin_bar = "======== $origin_title ========";

		$text  = "<<<<<<<< $custom_title <<<<<<<<\n";
		$text .= $custom;
		$text .= $origin_bar . "\n";
		$text .= $origin;
		$text .= str_repeat("=", strlen($origin_bar)) . "\n";
		$text .= $new;
		$text .= ">>>>>>>> $new_title >>>>>>>>\n";
	}

	return $text;
}

function format_diff_text($string, $wrap = true)
{
	if (trim($string) === '')
	{
		return '&nbsp;';
	}
	else
	{
		if ($wrap)
		{
			$string = nl2br(htmlspecialchars_uni($string));
			$string = preg_replace('#( ){2}#', '&nbsp; ', $string);
			$string = str_replace("\t", '&nbsp; &nbsp; ', $string);
			return "<code>$string</code>";
		}
		else
		{
			return '<pre style="display:inline">' . "\n" . htmlspecialchars_uni($string) . '</pre>';
		}
	}
}

/*=========================================================================*\
|| #######################################################################
|| # Downloaded: 04:56, Fri Sep 12th 2025
|| # CVS: $RCSfile$ - $Revision: 111950 $
|| #######################################################################
\*=========================================================================*/
